/***************************************************************************
 *
 * kernel_binfmt.c
 *
 * LOMAC - Low Water-Mark Mandatory Access Control for Linux
 * Copyright (c) 1999, 2000, 2001, 2002 Networks Associates
 * Technology, Inc.  All rights reserved.
 * 
 * This file is part of LOMAC.
 *
 * LOMAC is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * LOMAC is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with LOMAC; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *
 *
 * The sys_execve() operation loads a file into memory for execution.
 * This operation is effectively a read.  LOMAC must monitor this
 * read.  Unfortunately, the simple system-call-jump-table-patching
 * interception mechanism in intercept.h cannot intercept
 * sys_execve().  (Perhaps this inability is due to some stack
 * manipulation done in the architecture-specific assembly code?  I
 * don't really know.)  So, instead of intercepting execve's with the
 * intercept.h method, we register our own (fake) binary format
 * handler in the kernel's binary format list.
 *
 * The kernel keeps a linked list of binary format handlers.  Each
 * entry in the list contains a jump table of binary format handling
 * functions.  On each execve(), the kernel traverses the list,
 * calling each entry's jump-table function to handle the image
 * specified by the execve() call.  Each entry's function gives it a
 * shot.  If an entry decides it can't handle the image, it returns
 * -ENOEXEC and the kernel gives the next entry in line a try.
 *
 * Our binary format handler always returns -ENOEXEC, since it isn't a
 * real binary format handler at all.  Instead of trying to handle the
 * binary image, it examines the execve() parameters for file dentries
 * and calls monitor_read().
 *
 * This approach has (at least) one unaddressed fault:
 *
 * 1.  When we register our format, the kernel inserts it at the head
 *     of the binary format handler list.  Our entry must always be
 *     at the head of the list in order for it to be queried on every
 *     execve().  If someone registers a handler after we do, we
 *     will be bumped into second place.  Program loads done by the new
 *     first handler will not be monitored as reads.
 *
 ***************************************************************************/

#include <linux/kernel.h>   /* for KERN_INFO */
#include <linux/sched.h>    /* for `current' */
#include <linux/binfmts.h>  /* for struct linux_binfmt */

#include "kernel_interface.h"
#include "lomac_plm.h"
#include "lomac_monitor.h"


static int load_lomac_binary( struct linux_binprm *, struct pt_regs * );
static int load_lomac_library( int );

static struct linux_binfmt lomac_format = {
  NULL,
  NULL,
  load_lomac_binary,
  load_lomac_library,
  NULL
};

/* load_lomac_binary()
 *
 * in:     bprm - pointer to parameters specified by execve
 *         regs - register values
 * out:    nothing
 * return: -ENOEXEC
 *
 */

static int load_lomac_binary( struct linux_binprm *bprm,
			      struct pt_regs *regs ) {

  lattr_t program_lattr;  /* LOMAC attrs of program file being loaded */
  char *canabsname_s;     /* canonical absolute name of file being loaded */

  if( bprm && bprm->dentry ) {

    /* Set program file's level */
    if( !( canabsname_s = (char *)__get_free_page( GFP_KERNEL ) ) ) {
      free_page( (unsigned long)canabsname_s );
      return( -ENOEXEC );   /* no monitor, no exec. */
    }
    if( get_object_canabspath( bprm->dentry, canabsname_s, MAX_PATH_LEN ) ) {
      free_page( (unsigned long)canabsname_s );
      return( -ENOEXEC );   /* no monitor, no exec. */
    }
#ifdef PARANOID
    program_lattr.level = LOMAC_INVALID_LEVEL;
#endif
    get_pathname_lattr( canabsname_s, &program_lattr );
    free_page( (unsigned long)canabsname_s );
#ifdef PARANOID
    if( program_lattr.level == LOMAC_INVALID_LEVEL ) {
      PANIC( "LOMAC/K: kernel_binfmt found invalid program level.\n" );
    }
#endif
    set_object_lattr( bprm->dentry, program_lattr );

    /* monitor the read */
    monitor_read_object( current, bprm->dentry );

  }
  return( -ENOEXEC );  /* Nope, this isn't the correct binary format */

} /* load_lomac_binary() */


/* load_lomac_library()
 *
 * Should we also monitor this operation as a read? ##
 *
 */

static int load_lomac_library( int fd ) {
  return( -ENOEXEC );  /* Nope, this isn't the correct binary format */
}

/* ------------------------- Public Functions ------------------------- */

/* register_lomac_binfmt()
 *
 * in:     nothing
 * out:    nothing
 * return: nothing
 *
 * Adds `lomac_format' to the kernel's list of binary format handlers,
 * effectively allowing LOMAC to intercept sys_execve() calls.
 *
 */

void register_lomac_binfmt( void ) {

  if( register_binfmt( &lomac_format ) ) {
    panic( "LOMAC(lomac_binfmt.c) - failed to register binfmt" );
  }

} /* register_lomac_binfmt() */




