/************************************************************************
 *
 * kernel_driver.c
 *
 * LOMAC - Low Water-Mark Mandatory Access Control for Linux
 * Copyright (c) 1999, 2000, 2001, 2002 Networks Associates
 * Technology, Inc.  All rights reserved.
 * 
 * This file is part of LOMAC.
 *
 * LOMAC is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * LOMAC is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with LOMAC; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *
 *
 * Routines to manage LOMAC's user-space interface: the /dev/lomac
 * pseudodevice.  User-space utility programs use this device to
 * set the LOMAC policy, query the state of the LOMAC LKM, and
 * determine the LOMAC attributes of processes and files, etc.
 * during run-time.
 *
 *
 *
 ************************************************************************/

#include <asm/uaccess.h>    /* for verify_area() */

#include "kernel_util.h"
#include "lomac_log.h"
#include "lomac_attr.h"
#include "lomac_plm.h"

/* Commands for ioctl() interface. */
#define GET_PID_LEVEL_IOCTL_CMD  0
#define GET_FILE_LEVEL_IOCTL_CMD 1


int do_ioctl( struct inode *, struct file *, unsigned int, unsigned long );

/* This is the file operation jump table for the LOMAC *
 * character device.  We'll be using ioctl, only.      */
struct file_operations lomac_fops = { NULL, NULL, NULL, NULL, NULL,
				      do_ioctl, NULL, NULL, NULL, NULL,
				      NULL, NULL, NULL };


/* ----------------------- Private Functions ----------------------- */


/* do_ioctl()
 *
 * in:     p_inode_ignored - ignored
 *         p_file_ignored  - ignored
 *         command - GET_PID_LEVEL_IOCTL_CMD  - return level of pid in `arg'.
 *                   GET_FILE_LEVEL_IOCTL_CMD - return level of file named
 *                                              by the string `arg'.
 *         arg     - if `command' is GET_PID_LEVEL_IOCTL_CMD, then `arg' is
 *                   the pid of the process whose level we want to query.
 *                   if `command' is GET_FILE_LEVEL_IOCTL_CMD, then `cmd'
 *                   points to the name of the file whose level we want
 *                   to query, stored as a null-terminated string.
 * out:    nothing
 * return: the level of the process or file indicated by `command' and `arg',
 *         or -1 if no such process or file exists.  Also returns -1 on
 *         a bad `cmd' parameter.
 *
 *     User-space utility programs may query the state of the LOMAC LKM
 * through this ioctl() interface.
 *
 */

int do_ioctl( struct inode *p_inode_ignored, struct file *p_file_ignored,
	      unsigned int command, unsigned long arg ) {

  lattr_t lattr;       /* holds attrs of process or file indicated by `arg' */
  unsigned int filename_length;       /* length of file name from userspace */
  struct task_struct *p_proc;  /* used to find process table entries by pid */
  struct dentry *p_dentry;                    /* dentry for files, ignored. */
  struct dentry *p_dir_dentry;           /* entry for parent dirs, ignored. */
  char *path_s;                    /* kernel copy of paths supplied by user */
  int ret_val;                              /* holds function return values */

  switch( command ) {
  case GET_PID_LEVEL_IOCTL_CMD:          /* look up process level */
    
    /* `arg' contains the pid of the process whose level we want *
     * to know.  Find the task_struct corresponding to `pid'.    */
    lattr.level = LOMAC_INVALID_LEVEL;
    p_proc = current;
    do {
      if( p_proc->pid == arg ) {
	get_subject_lattr( p_proc, &lattr );
	break;
      }
    } while( ( p_proc = p_proc->next_task ) != current );
    
    if( lattr.level == LOMAC_INVALID_LEVEL ) {
      return( -1 );
    }
    
    break;
    
  case GET_FILE_LEVEL_IOCTL_CMD:         /* look up file level */
    
    /* When a file's attrs are queried, we expect `arg' to point to a  *
     * variable-sized block of user-space memory organized as follows: *
     * First, an unsigned int containing the size of a following       *
     * buffer, and second, the buffer itself.  The buffer is used to   *
     * contain the absolute path to the file in question, as a         *
     * null-terminated string.                                         */
    
    /* First, verify that we can read the string-length uint, *
     * and copy it into kernel memory as `filename_length'.   */
    if( verify_area( VERIFY_READ, (void *)arg, sizeof( unsigned int ) ) ) {
      return( -1 );
    }
    copy_from_user( &filename_length, (void *)arg, sizeof( int ) );
    
    arg = (unsigned int)((unsigned int *)arg + 1);  /* skip over size uint */

    /* Convert to an absolute path in canonical form. */
    path_s = getname( (char *)arg );
    if( IS_ERR( path_s ) ) {
      return( PTR_ERR( path_s ) );
    }
    ret_val = make_canabspath( path_s, log_s, &p_dir_dentry, &p_dentry );
    putname( path_s );
    if( p_dir_dentry ) {
      dput( p_dir_dentry );
    }
    if( p_dentry ) {
      dput( p_dentry );
    }
    if( 0 > ret_val ) {
      /* unable to convert to canabspath.  Return error value. */
      return( ret_val );
    }
    
#ifdef PARANOID
    /* verify that `log_s' is an absolute path in canonical form. */
    if( !is_canabspath( log_s ) ) {
      PANIC( "LOMAC/K: do_ioctl() failed to created faulty canabspath.\n" );
    }
#endif

    /* We're determining the attrs by checking the PLM, not the actual *
     * attrs stored in the inode structure.  We do this because it's   *
     * easier, and if LOMAC is working correctly, it will always       *
     * deliver the correct result.                                     */
    get_pathname_lattr( log_s, &lattr );
    
#ifdef PARANOID
    if( lattr.level == LOMAC_INVALID_LEVEL ) {
      log_print();  /* should print out log_s, containing the offending path */
      PANIC( "LOMAC/K: do_ioctl() returned invalid level (path in log).\n" );
    }
#endif
    
    break;
    
  default:                               /* error - bad cmd value */
    
    /* Allessandro Rubini's Linux Drivers book says that POSIX says  *
     * that I should return -ENOTTY here for backward compatibility. */
    return( -EINVAL );
    
  } /* switch( command ) */

  return( lattr.level );

} /* do_ioctl() */


/* ------------------------ Public Functions ----------------------- */


/* init_driver()
 *
 * in:     nothing
 * out:    nothing
 * return: nothing
 *
 */

void init_driver( void ) {

  /* Register a character device for LOMAC.  The 0 in the first parm  *
   * indicates that we want the kernel to dynamically allocate a free *
   * device major number for us.  User-space programs will be able to *
   * determine our device major number by examining the file          *
   * /proc/devices.  This will fail only if there are no free major   *
   * device numbers left for the kernel to allocate.                  */

  if( !register_chrdev( 0, "lomac", &lomac_fops ) ) {
    PANIC( "LOMAC/K: no major device number available for lomac" );
  }

} /* init_driver() */
