#ifndef _LINUX_INTERCEPT_H_
#define _LINUX_INTERCEPT_H_

/****************************************************************************
 *
 * linux_intercept.h
 *
 * LOMAC - Low Water-Mark Mandatory Access Control for Linux
 * Copyright (c) 1999, 2000, 2001, 2002 Networks Associates
 * Technology, Inc.  All rights reserved.
 * 
 * This file is part of LOMAC.
 *
 * LOMAC is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * LOMAC is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with LOMAC; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *
 * This file contains macros which encapsulate the details of "intercepting"
 * system calls.  We "intercept" a system call by replacing the pointer to
 * the kernel's system call routine in the `sys_call_table' with a pointer
 * to a new function of our own devising.  This causes the kernel's system
 * call dispatcher to call our new function rather than the original one
 * whenever a process invokes that system call.
 *
 * 
 *
 ****************************************************************************/

#include <linux/unistd.h>      /* for system call name->number mapping */

/* `sys_call_table' is the kernel's system call table - an array of    *
 * function pointers.  A system call's number is the index of its      *
 * function in the array.  See entry.S in arch-specific assembly area. */
extern void *sys_call_table[];

/* This macro defines a variable to store the original function      *
 * address for the syscall named by `syscall_name'.  When we replace *
 * this syscall later on with the REPLACE_SYSCALL macro, we'll use   *
 * this variable to remember the replaced function address so we can *
 * eventually restore it with the RESTORE_SYSCALL macro.  Make sure  *
 * you use this macro in a scope which is visible from the places    *
 * where you use the REPLACE/RESTORE macros.                         */

#define SETUP_SYSCALL_STORAGE( syscall_name ) \
   void *orig_ ## syscall_name

/* Use the REPLACE_SYSCALL macro to make the syscall named by  *
 * `syscall_name' correspond to the function `replacement_fxn' */

#define REPLACE_SYSCALL( syscall_name, replacement_fxn ) \
   orig_ ## syscall_name = sys_call_table[ __NR_ ## syscall_name ]; \
   sys_call_table[ __NR_ ## syscall_name ] = (void *)replacement_fxn

/* Use the RESTORE_SYSCALL macro to undo the effects of REPLACE_SYSCALL */

#define RESTORE_SYSCALL( syscall_name ) \
   sys_call_table[ __NR_ ## syscall_name ] = orig_ ## syscall_name

#endif



