/***************************************************************************
 *
 * kernel_interface.c
 *
 * LOMAC - Low Water-Mark Mandatory Access Control for Linux
 * Copyright (c) 1999, 2000, 2001, 2002 Networks Associates
 * Technology, Inc.  All rights reserved.
 * 
 * This file is part of LOMAC.
 *
 * LOMAC is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * LOMAC is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with LOMAC; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *
 * This file contains routines that provide LOMAC's high-level
 * routines with a kernel-independent interface to several important
 * kinds of kernel objects.  This level of abstraction is of practical
 * importance to LOMAC, because ongoing kernel developments change the
 * way some kernel objects are accessed over time.  When such changes
 * occur, this file can encapsulate the change, leaving the
 * higher-level routines as they were.
 *
 ***************************************************************************/

#include "kernel_interface.h"
#include "kernel_util.h"
#include "lomac_attr.h"
#include "lomac_log.h"


/***********************************************************************
 * Types and routines to map the LOMAC notion of "subject" onto the
 * kernel notion of process.
 ***********************************************************************/

/* The following constants are meant to coexist with the PF_* constants *
 * defined in sched.h.  All these flags are meant for task_struct's     *
 * `flags' field.                                                       */
#define SUBJECT_LEVEL_1    0x00010000
#define SUBJECT_LEVEL_2    0x00020000
#define SUBJECT_LEVEL_BITS ( SUBJECT_LEVEL_1 | SUBJECT_LEVEL_2 )
#define SUBJECT_CATEGORY_1 0x01000000
#define SUBJECT_CATEGORY_2 0x02000000
#define SUBJECT_CATEGORY_3 0x04000000
#define SUBJECT_CATEGORY_4 0x08000000
#define SUBJECT_CATEGORY_BITS ( SUBJECT_CATEGORY_1 | SUBJECT_CATEGORY_2 | \
                                SUBJECT_CATEGORY_3 | SUBJECT_CATEGORY_4 )


/* get_subject_id()
 * 
 * in:     p_subject - pointer to subject whose ID we wish to know
 *         id_s      - pointer to a character buffer to hold ID string.
 *         bufflen   - number of chars available for non-terminator
 *                     chars in `id_s'.
 * out:    id_s      - buffer will contain null-terminated ID string
 * return: the number of non-terminator characters written to `id_s'.
 *
 *     This function writes a string describing the identity of
 * the subject indicated by `p_subject' to `id_s'.  This string
 * is guaranteed to be no more than `bufflen' non-terminator characters
 * long.
 *
 * Currently the ID string is a concatenation of the following process
 * attributes: pid, pgrp, uid, and the name of the executable the 
 * process is running.  This string is for logging purposes only; 
 * higher-level LOMAC functions do not interpret the contents of this
 * string.
 *
 */

int get_subject_id( const lomac_subject_t *p_subject, char *id_s, 
		    int bufflen ) {

  char ascii[ MAX_DIGITS ];  /* holds the ASCII representation of ints */
  int component_length;      /* length of component parts of ID string */
  int space_left;            /* portion of `bufflen' used so far... */

#ifdef PARANOID
  if( !p_subject ) {
    PANIC( "LOMAC/K: null subject pointer passed to get_subject_id()\n" );
  }
  if( !id_s ) {
    PANIC( "LOMAC/K: null id_s pointer passed to get_subject_id()\n" );
  }
  if( bufflen <= 0 ) {
    PANIC( "LOMAC/K: non-positive buffer length "
	   "passed to get_subject_id()\n" );
  }
#endif


  space_left = bufflen;

  /* Append pid. */
  if( space_left < 1 ) {
    goto out;
  }
  STRNCAT( id_s, "p", 1 );
  space_left--;
  TOASCII( p_subject->pid, ascii );
  component_length = STRNLEN( ascii, MAX_DIGITS );
  if( component_length > space_left ) {
    goto out;
  }
  STRNCAT( id_s, ascii, component_length );
  space_left -= component_length;

  /* Append pgid. */
  if( space_left < 1 ) {
    goto out;
  }
  STRNCAT( id_s, "g", 1 );
  space_left--;
  TOASCII( p_subject->pgrp, ascii );
  component_length = STRNLEN( ascii, MAX_DIGITS );
  if( component_length > space_left ) {
    goto out;
  }
  STRNCAT( id_s, ascii, component_length );
  space_left -= component_length;

  /* Append uid. */
  if( space_left < 1 ) {
    goto out;
  }
  STRNCAT( id_s, "u", 1 );
  space_left--;
  TOASCII( p_subject->uid, ascii );
  component_length = STRNLEN( ascii, MAX_DIGITS );
  if( component_length > space_left ) {
    goto out;
  }
  STRNCAT( id_s, ascii, component_length );
  space_left -= component_length;

  /* Append program name.  We use the comm field of task_struct to get *
   * the program name.  This gives us only a limited number of         *
   * characters.  The PROC_PID_EXE case in linux/fs/proc/link.c shows  *
   * how to get the actual dentry for the exe file.  Perhaps it would  *
   * be better to use this method in the future.                       */
  if( space_left < 1 ) {
    goto out;
  }
  STRNCAT( id_s, ":", 1 );
  space_left--;
  component_length = STRNLEN( p_subject->comm, space_left );
  STRNCAT( id_s, p_subject->comm, component_length );
  space_left -= component_length;

 out:

#ifdef PARANOID

  /* We should have used up to `bufflen' chars, but no more. */
  if( ( bufflen - space_left ) < 0 ) {
    PANIC( "LOMAC/K: id too long in get_subject_id()\n" );
  }

  /* All chars from `id_s[ 0 ]' to `id_s[ bufflen - space_left - 1 ]' *
   * should be non-null.  `id_s[ bufflen - space_left ]' should be    *
   * null.  We're re-using component_length here as an index offset   *
   * counter just to avoid declaring another variable.                */
  for( component_length = 0; component_length < ( bufflen - space_left );
       component_length++ ) {
    if( id_s[ component_length ] == '\0' ) {
      PANIC( "LOMAC/K: prematurely terminated id in get_subject_id()\n" );
    }
  }
  if( id_s[ bufflen - space_left ] != '\0' ) {
      PANIC( "LOMAC/K: improperly terminated id in get_subject_id()\n" );
  }

#endif

  return( bufflen - space_left );

} /* get_subject_id() */


/* set_subject_lattr()
 * 
 * in:     p_subject - pointer to subject whose attributes we want to set
 *         lattr     - attribute information we desire to set
 * out:    see description of side-effects, below
 * return: nothing
 *
 * This routine sets the LOMAC attributes of `p_subject' to `lattr'.
 *
 * This version of LOMAC stores subject attribute information by shoe-horning
 * its attribute bits into an unused part of the struct task_struct's
 * `flags' field.  
 */

void set_subject_lattr( lomac_subject_t *p_subject, lattr_t lattr ) {

#ifdef PARANOID
  if( !p_subject ) {
    PANIC( "LOMAC/K: null subject pointer passed to set_subject_lattr()\n" );
  }
#endif

  /* Set level bits.  A subject can be at only one level at a time. */
  switch( lattr.level ) {
  case LOMAC_HIGHEST_LEVEL:
    p_subject->flags &= ~SUBJECT_LEVEL_BITS;
    p_subject->flags |= SUBJECT_LEVEL_2;
    break;
  case LOMAC_LOWEST_LEVEL:
    p_subject->flags &= ~SUBJECT_LEVEL_BITS;
    p_subject->flags |= SUBJECT_LEVEL_1;
    break;
  default:
#ifdef PARANOID
    printk( "LOMAC/K: attempted to set invalid level for p%ug%u\n", 
	    p_subject->pid, p_subject->pgrp );
    PANIC( "LOMAC/K: attempted to set invalid subject level.\n" );
#endif /* PARANOID */
  }

  /* Set category bits.  A subject can have many category bits set at once. */
  p_subject->flags &= ~SUBJECT_CATEGORY_BITS;
  if( lattr.categories & LOMAC_CATEGORY_ONE ) {
    p_subject->flags |= SUBJECT_CATEGORY_1;
  }
  if( lattr.categories & LOMAC_CATEGORY_TWO ) {
    p_subject->flags |= SUBJECT_CATEGORY_2;
  }
  if( lattr.categories & LOMAC_CATEGORY_THREE ) {
    p_subject->flags |= SUBJECT_CATEGORY_3;
  }
  if( lattr.categories & LOMAC_CATEGORY_FOUR ) {
    p_subject->flags |= SUBJECT_CATEGORY_4;
  }

} /* set_subject_lattr() */


/* get_subject_lattr()
 *
 * in:     p_subject - pointer to subject whose attributes we wish to know
 * out:    p_lattr   - attributes of `p_subject' will be written here
 * return: nothing
 *
 * This function reads the architecture-specific LOMAC attributes bits
 * from `p_subject' and fills in the corresponding architecture-independent
 * bits in `p_lattr'.
 *
 */

void get_subject_lattr( const lomac_subject_t *p_subject, lattr_t *p_lattr ) {

#ifdef PARANOID
  if( !p_subject ) {
    PANIC( "LOMAC/K: null subject pointer passed to get_subject_lattr()\n" );
  }
  if( !p_subject ) {
    PANIC( "LOMAC/K: null attribute pointer passed to get_subject_lattr()\n" );
  }
#endif
  
  /* Get level attribtues */
  switch( p_subject->flags & SUBJECT_LEVEL_BITS ) {
  case SUBJECT_LEVEL_2:
    p_lattr->level = LOMAC_HIGHEST_LEVEL;
    break;
  case SUBJECT_LEVEL_1:
    p_lattr->level = LOMAC_LOWEST_LEVEL;
    break;
  default:
    p_lattr->level = LOMAC_INVALID_LEVEL;
#ifdef PARANOID
    printk( "LOMAC/K: found invalid level for p%ug%u\n", 
	    p_subject->pid, p_subject->pgrp );
#endif
  }
  
  /* Get category attributes */
  p_lattr->categories = 0;
  if( p_subject->flags & SUBJECT_CATEGORY_1 ) {
    p_lattr->categories |= LOMAC_CATEGORY_ONE; 
  }
  if( p_subject->flags & SUBJECT_CATEGORY_2 ) {
    p_lattr->categories |= LOMAC_CATEGORY_TWO; 
  }
  if( p_subject->flags & SUBJECT_CATEGORY_3 ) {
    p_lattr->categories |= LOMAC_CATEGORY_THREE; 
  }
  if( p_subject->flags & SUBJECT_CATEGORY_4 ) {
    p_lattr->categories |= LOMAC_CATEGORY_FOUR; 
  }
  
} /* get_subject_lattr() */



/***********************************************************************
 * Types and routines to map the LOMAC notion of "object" to the
 * kernel notion of dcache entries.
 ***********************************************************************/

/* The following constants are meant to coexist with the DCACHE flags  *
 * defined in dcache.h.  These flags are all supposed to reside in the *
 * d_flags fields of the dentry structure.                             */

#define OBJECT_LEVEL_1    0x00010000
#define OBJECT_LEVEL_2    0x00020000
#define OBJECT_LEVEL_BITS ( OBJECT_LEVEL_1 | OBJECT_LEVEL_2 )
#define OBJECT_CATEGORY_1 0x00100000
#define OBJECT_CATEGORY_2 0x00200000
#define OBJECT_CATEGORY_3 0x00300000
#define OBJECT_CATEGORY_4 0x00400000
#define OBJECT_CATEGORY_BITS ( OBJECT_CATEGORY_1 | OBJECT_CATEGORY_2 | \
                               OBJECT_CATEGORY_3 | OBJECT_CATEGORY_4 )


/* get_object_id()
 *
 * in:     p_object - pointer to object whose identity we wish to know
 *         id_s     - buffer to contain path corresponding to `p_object'.
 *         bufflen  - length of `id_s' buffer.
 * out:    id_s     - will contain absolute canonical path for `p_object',
 *                    or a string constructed from its device and inode
 *                    numbers if there is no path corresponding to 
 *                    `p_object'. 
 * return: number of non-terminator characters written to `id_s'.
 *
 *     This function appends a string identifying `p_object' to
 * `id_s'.  The string will either be an absolute canonical path, or
 * for those objects that have no such path, it will be a
 * concatenation of device major number, device minor number, and
 * inode number.  This string is for logging purposes only;
 * higher-level LOMAC functions do not interpret the contents of this
 * string.
 *
 */

int get_object_id( const lomac_object_t *p_object, char *id_s, 
		   int bufflen ) {

#ifdef PARANOID
  if( !p_object ) {
    PANIC( "LOMAC/K: null object pointer passed to get_object_id()\n" );
  }
  if( !id_s ) {
    PANIC( "LOMAC/K: null id_s pointer passed to get_object_id()\n" );
  }
  if( bufflen <= 0 ) {
    PANIC( "LOMAC/K: non-positive buffer length passed to get_object_id()\n" );
  }
#endif


  /* If the dcache contains a path, then copy it to `id_s', otherwise *
   * construct an id string from the corresponding inode's device     *
   * and inode numbers.                                               *
   * We shouldn't need the IS_PIPE thing, but it seems that linux     *
   * pipes have paths of "/" in the dcache.  Go figure.               *
   * Same goes for IS_SOCKET.                                         */

  if( IS_PIPE( p_object ) || IS_SOCKET( p_object ) ||
      get_object_canabspath( p_object, id_s, bufflen ) ) {

    /* If there isn't enough room for a maximally long ID, just give up. */
    if( bufflen < MAX_DIGITS * 3 ) {
      return( 0 );
    }

    /* No path for this object, make up an id based on numbers... */
    sprintf( id_s, "D%ud%ui%lu", 
	     MAJOR(p_object->d_inode->i_dev), 
	     MINOR(p_object->d_inode->i_dev),
	     p_object->d_inode->i_ino );

  }

#ifdef PARANOID
  /* We should have used no more than `bufflen-1' non-null characters. */
  if( STRNLEN( id_s, bufflen ) >= bufflen ) {
    PANIC( "LOMAC/K: object id too long in get_object_id()\n" );
  }
#endif

  return( STRNLEN( id_s, bufflen ) );

} /* get_object_id() */


/* get_object_canabspath()
 *
 * in:     p_object - pointer to object whose canabspath we want.
 *         path_s   - buffer to contain path corresponding to `p_object'.
 *         bufflen  - length of `path_s' buffer.
 * out:    path_s   - will contain absolute canonical path for `p_object'
 *                    if such a path exists.  If there is no such path,
 *                    `path_s' will not be modified.
 * return: value    condition
 *         -----    ---------
 *           0      `path_s' contains the canabspath of `p_object'.
 *           1      there is no canabspath for `p_object'.
 *
 *  This function is used to get the canonical absolute paths of
 *  objects, in order to facilitate lookups in the file system object
 *  -> level mapping.  This function works only on objects which
 *  exist.  
 *
 */

int get_object_canabspath( const lomac_object_t *p_object, char *path_s,
			   int bufflen ) {

#ifdef PARANOID
  if( !p_object ) {
    PANIC( "LOMAC/K: null object pointer "
	   "passed to get_object_canabspath()\n" );
  }
  if( !path_s ) {
    PANIC( "LOMAC/K: null path_s pointer "
	   "passed to get_object_canabspath()\n" );
  }
  if( bufflen <= 0 ) {
    PANIC( "LOMAC/K: non-positive buffer "
	   "length passed to get_object_canabspath()\n" );
  }
#endif

  if( p_object->d_name.name && p_object->d_name.name[ 0 ] != '\0' ) {

    strncpy( path_s, d_path( (struct dentry *)p_object, path_s, bufflen ),
	     bufflen );  /* the cast is to remove const */
    return( 0 );

  } else {

    return( 1 );

  }

} /* get_object_canabspath() */


/* set_object_lattr()
 *
 * in:     p_object - object whose LOMAC attributes we want to set.
 *         lattr    - new LOMAC attributes for p_object.
 * out:    p_object - bits will be twiddled to reflect new attributes.
 * return: nothing
 * 
 * This version of LOMAC stores object levels by making use of some
 * extra bits in the `d_flags' field of struct dentry.
 *
 */

void set_object_lattr( lomac_object_t *p_object, lattr_t lattr ) {

#ifdef PARANOID
  if( !p_object ) {
    PANIC( "LOMAC/K: null object pointer passed to set_object_lattr()\n" );
  }
#endif

  /* Set level.  An object can be at only one level. */
  switch( lattr.level ) {
  case LOMAC_HIGHEST_LEVEL:
    p_object->d_flags &= ~OBJECT_LEVEL_BITS;
    p_object->d_flags |= OBJECT_LEVEL_2;
    break;
  case LOMAC_LOWEST_LEVEL:
    p_object->d_flags &= ~OBJECT_LEVEL_BITS;
    p_object->d_flags |= OBJECT_LEVEL_1;
    break;
  default:
#ifdef PARANOID
    printk( "LOMAC/K: attempted to set invalid level for D%ud%ui%lu", 
	    MAJOR(p_object->d_inode->i_dev),
	    MINOR(p_object->d_inode->i_dev),
	    p_object->d_inode->i_ino );
#endif /* PARANOID */
  }

  /* Set category attributes.  An object can have many categories. */
  p_object->d_flags &= ~OBJECT_CATEGORY_BITS;
  if( lattr.categories & LOMAC_CATEGORY_ONE ) {
    p_object->d_flags |= OBJECT_CATEGORY_1;
  }  
  if( lattr.categories & LOMAC_CATEGORY_TWO ) {
    p_object->d_flags |= OBJECT_CATEGORY_2;
  }  
  if( lattr.categories & LOMAC_CATEGORY_THREE ) {
    p_object->d_flags |= OBJECT_CATEGORY_3;
  }  
  if( lattr.categories & LOMAC_CATEGORY_FOUR ) {
    p_object->d_flags |= OBJECT_CATEGORY_4;
  }  

} /* set_object_lattr() */


/* get_object_lattr()
 *
 * in:     p_object - the object whose LOMAC attributes we wish to know.
 * out:    p_lattr  - `p_object's attributes will go here.
 * return: nothing
 *
 * This function reads the architecture-specific LOMAC attributes bits
 * from `p_object' and fills in the corresponding architecture-independent
 * bits in `p_lattr'.
 *
 */

void get_object_lattr( const lomac_object_t *p_object, lattr_t *p_lattr ) {

#ifdef PARANOID
  if( !p_object ) {
    PANIC( "LOMAC/K: null object pointer passed to get_object_lattr()\n" );
  }
  if( !p_lattr ) {
    PANIC( "LOMAC/K: null lattr pointer passed to get_object_lattr()\n" );
  }
#endif

  switch( p_object->d_flags & OBJECT_LEVEL_BITS ) {
  case OBJECT_LEVEL_2:
    p_lattr->level = LOMAC_HIGHEST_LEVEL;
    break;
  case OBJECT_LEVEL_1:
    p_lattr->level = LOMAC_LOWEST_LEVEL;
    break;
  default:
    p_lattr->level = LOMAC_INVALID_LEVEL;
  }

  /* Get object categories */
  p_lattr->categories = 0;
  if( p_object->d_flags & OBJECT_CATEGORY_1 ) {
    p_lattr->categories |= LOMAC_CATEGORY_ONE; 
  }  
  if( p_object->d_flags & OBJECT_CATEGORY_2 ) {
    p_lattr->categories |= LOMAC_CATEGORY_TWO; 
  }  
  if( p_object->d_flags & OBJECT_CATEGORY_3 ) {
    p_lattr->categories |= LOMAC_CATEGORY_THREE; 
  }  
  if( p_object->d_flags & OBJECT_CATEGORY_4 ) {
    p_lattr->categories |= LOMAC_CATEGORY_FOUR;
  }  

} /* get_object_level() */


/* get_subject_program_object()
 *
 * in:     p_subject - subject whose program object we want
 * out:    nothing
 * return: object `p_subject' is using for program text, or NULL on error.
 *
 * This function effectively returns the dentry * of the file the process
 * is using for program text.  The code in the PROC_PID_EXE case of
 * proc_follow_link() in linux/fs/proc/link.c does something similar
 * in a different way.
 *
 * When this function returns an object (dentry), it will increment
 * that object's reference count.  The caller must eventually call
 * object_done() on that object.
 *
 */

lomac_object_t *get_subject_program_object( lomac_subject_t *p_subject ) {

  struct vm_area_struct *p_vma; /* used to iterate through list of vm areas */

  if( !p_subject->mm ) {
    return( NULL );
  }

  /* Iterate through `p_subject's list of vm areas looking *
   * for the one that describes `p_subject's program text. */
  for( p_vma = p_subject->mm->mmap; p_vma; p_vma = p_vma->vm_next ) {
    if( p_vma->vm_file && ( p_vma->vm_flags & VM_EXECUTABLE ) ) {
      return( dget( p_vma->vm_file->f_dentry ) );
    }
  }

  return( NULL );

} /* get_subject_program_object() */


/* path_to_object()
 *
 * in:     path_s - path whose object we wish to find (a kernel-space string)
 * out:    nothing
 * return: pointer to the object corresponding to `path_s', or NULL if
 *         no such object exists. 
 *
 * This function is an interface to kernel_namei(), which is an
 * interface to the kernel's namei() function that takes its argument
 * from kernel-space rather than user-space.
 *
 * This function has simplified return value behavior - it returns
 * NULL if no object could be found, for any reason.
 *
 * If the object corresponding to `path_s' is found, namei() will
 * increment its reference count.  So, the caller must eventually
 * call object_done() on that object.
 *
 */

lomac_object_t *path_to_object( const char *path_s ) {

  lomac_object_t *p_object;   /* the object, if found */

  if( IS_ERR( p_object = kernel_namei( path_s ) ) ) {
    return( NULL );
  }

  return( p_object );

} /* path_to_object() */


/* object_done()
 *
 * in:     p_object - object we are done manipulating
 * out:    p_object - will have reference count decreased
 * return: nothing
 *
 * This is an interface to dput()
 * */

void object_done( lomac_object_t *p_object ) {
  dput( p_object );
}



