#ifndef _KERNEL_INTERFACE_H_
#define _KERNEL_INTERFACE_H_

/***************************************************************************
 *
 * kernel_interface.h
 *
 * LOMAC - Low Water-Mark Mandatory Access Control for Linux
 * Copyright (c) 1999, 2000, 2001, 2002 Networks Associates
 * Technology, Inc.  All rights reserved.
 * 
 * This file is part of LOMAC.
 *
 * LOMAC is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * LOMAC is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with LOMAC; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *
 * This file is the public interface to kernel_interface.c
 *
 *
 ***************************************************************************/

#include <linux/sched.h>      /* for struct task_struct */
#include <linux/fs.h>         /* for struct inode and device/inode # macros */
#include <linux/dcache.h>     /* for struct dentry */
#include <linux/netdevice.h>  /* for struct device (the NIC structure) */

#include "lomac_attr.h"

/* LOMAC maps its notion of subject onto the kernel's task_struct *
 * (process) abstraction.                                         */
typedef struct task_struct lomac_subject_t;

int  get_subject_id( const lomac_subject_t *p_subject, char *id_s,
		     int bufflen );
void set_subject_lattr( lomac_subject_t *p_subject, lattr_t lattr );
void get_subject_lattr( const lomac_subject_t *p_subject, lattr_t *p_lattr );


typedef struct dentry lomac_object_t;

int  get_object_id( const lomac_object_t *p_object, char *id_s,
		    int bufflen );
int  get_object_canabspath( const lomac_object_t *p_object, char *path_s,
			    int bufflen );
void set_object_lattr( lomac_object_t *p_object, lattr_t lattr );
void get_object_lattr( const lomac_object_t *p_object, lattr_t *p_lattr );


/* These functions find objects based on certain criteria, such as  *
 * path.  They require the caller to invoke object_done() when they *
 * are finished with the found objects.                             */
lomac_object_t *get_subject_program_object( lomac_subject_t *p_subject );
lomac_object_t *path_to_object( const char *path_s );
void object_done( lomac_object_t *p_object );


/* Here are some types and routines to handle NICs.  LOMAC determines *
 * the level of network sockets according to the NIC they have read   *
 * packets from.                                                      */
typedef struct device lomac_nic_t;


/* This macro must take a string arg and print it to the system log. */
#define LOG( str )( printk( (str) ) )
/* hack - we use this prefix to set kernel log verbosity */
#define LOG_PREFIX KERN_INFO

/* This macro must print a message to the system console *
 * and panic the kernel.                                 */
#ifdef PARANOID
extern char *global_wrapper_name_s;
#define PANIC(s) printk( "LOMAC PANIC while handling %s\n", \
                         global_wrapper_name_s ); \
                 panic(s);
#else
#define PANIC panic
#endif /* if/else PARANOID */

/* These macro must present various libc str...(3) behaviors. */
#define STRCMP  strcmp
#define STRNCAT strncat
#define STRNLEN strnlen

/* This macro must place the decimal ASCII representation of *
 * `data' into `ascii'.                                      */
#define TOASCII( data, ascii )( sprintf( (ascii), "%u", (data) ) )

/* MAX_DIGITS is a sufficient number of decimal digits to describe a *
 * 64-bit uint with a null terminator, rounded to a multiple of 8.   */
#define MAX_DIGITS 24 

/* MAX_PATH_LEN is the length of the longest possible path string, *
 * plus 1 for a null terminator.                                   */
#define MAX_PATH_LEN PATH_MAX

/* LOG_BUFFER_LENGTH is a string length "large enough" to hold        *
 * interesting log messages.  You can set this to whatever you        *
 * want, the log routines will be careful not to exceed this limit.   *
 * Since some routines use log_s as scratch path space, this constant *
 * should always be at least MAX_PATH_LEN long.                       */
#define LOG_BUFFER_LENGTH MAX_PATH_LEN +1


#endif






