/***************************************************************************
 *
 * kernel_lkm.c
 *
 * LOMAC - Low Water-Mark Mandatory Access Control for Linux
 * Copyright (c) 1999, 2000, 2001, 2002 Networks Associates
 * Technology, Inc.  All rights reserved.
 * 
 * This file is part of LOMAC.
 *
 * LOMAC is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * LOMAC is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with LOMAC; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *
 *
 * This file contains the code needed to load, initialize, and unload the
 * lomac Loadable Kernel Module (LKM).
 *
 * I suspect that the grandfathering procedure is not atomic, even
 * when logging is turned off.  Consequently, new processes can enter
 * the process table and existing processes can open new files during
 * the grandfathering procedure, and the grandfathering procedure will
 * miss these entrances/opens.  This hasn't caused a problem at this
 * point, probably because LOMAC sets up its interception of system
 * calls before it begins grandfathering, so all the operations that
 * are interleaved with the grandfathering procedure are handled as
 * they would be during LOMAC run-time.  Still, this situation seems
 * sloppy, and should be cleaned up in a manner that provides an
 * easily verifiable argument that all subjects and objects have been
 * givien initial levels at LOMAC load time.
 *
 ***************************************************************************/

#include <linux/module.h>  /* for module macros */

#include "kernel_interface.h"
#include "kernel_util.h"
#include "kernel_wrappers.h"
#include "kernel_driver.h"
#include "kernel_binfmt.h"
#include "lomac_log.h"
#include "lomac_plm.h"

/* This string puts the version number in the binary. */
static char *banner_s = 
"LOMAC v1.1.2 for Linux Copyright (C) 1999-2002 Networks Associates\n"
"Technology, Inc.  LOMAC comes with ABSOLUTELY NO WARRANTY.";

void handle_grandfathering( void );
void grandfather_pipe( struct task_struct *p_proc, struct dentry *p_pipe );
void grandfather_named_object( struct task_struct *p_proc,
			       struct dentry *p_object );
void grandfather_unix_socket( struct task_struct *p_proc,
			      struct dentry *p_socket );


/* init_module()
 *
 * in:     nothing
 * out:    nothing
 * return: value      condition
 *         -----      ---------
 *         0          success
 *
 * This function is called by the kernel when the lomac lkm is loaded.  It
 * initializes all of the lomac data structures, and patches the kernel's
 * system call vector to allow the lomac lkm to examine calls and perform
 * access control mediation on them.
 */

int init_module( void ) {

  /* Set up interception of the system calls which are interesting to Lomac. */
  REPLACE_SYSCALL( open,       wrap_open ); 
  REPLACE_SYSCALL( read,       wrap_read );
  REPLACE_SYSCALL( readv,      wrap_readv );
  REPLACE_SYSCALL( write,      wrap_write );
  REPLACE_SYSCALL( writev,     wrap_writev );
  REPLACE_SYSCALL( pipe,       wrap_pipe );
  REPLACE_SYSCALL( mknod,      wrap_mknod );
  REPLACE_SYSCALL( truncate,   wrap_truncate );
  REPLACE_SYSCALL( ftruncate,  wrap_ftruncate );
  REPLACE_SYSCALL( unlink,     wrap_unlink );
  REPLACE_SYSCALL( rename,     wrap_rename );
  REPLACE_SYSCALL( link,       wrap_link ); 
  REPLACE_SYSCALL( socketcall, wrap_socketcall );
  REPLACE_SYSCALL( kill,       wrap_kill );
  REPLACE_SYSCALL( mkdir,      wrap_mkdir );
  REPLACE_SYSCALL( rmdir,      wrap_rmdir );
  REPLACE_SYSCALL( mount,      wrap_mount );
  REPLACE_SYSCALL( umount,     wrap_umount );
  /*  REPLACE_SYSCALL( reboot,     wrap_reboot ); */
  REPLACE_SYSCALL( create_module,  wrap_create_module );
  REPLACE_SYSCALL( init_module,    wrap_init_module );
  REPLACE_SYSCALL( delete_module,  wrap_delete_module );

  /* initialize various data structures */
  handle_grandfathering();

  /* initialize driver (user-space info interface) */
  init_driver();

  /* initialize LOMAC's fake binary format for intercepting execve's */
  register_lomac_binfmt();

  /* Print a victory message to the console, indicating a successful load. */
  printk( "loaded %s\n", banner_s ); 
  return( 0 );

} /* init_module() */



/* cleanup_module()
 * 
 * in:     nothing
 * out:    nothing
 * return: nothing
 *
 * This function is called by the kernel in response to a request to unload
 * the lomac lkm.  Someday, this function should unpatch the system call
 * vector so the kernel can keep running once the LKM is gone, but this
 * requires some method to rescue processes that are currently blocked
 * in lomac-intercepted system calls.
 */

void cleanup_module( void ) {

  /* Remove interceptions; restore `sys_call_table' to it's original state. */
  /* RESTORE_SYSCALL( ... ); */

  printk( "unloaded %s\n", banner_s );

} /* cleanup_module() */


/* handle_grandfathering()
 * 
 * in:     nothing
 * out:    nothing
 * return: nothing
 *
 *     Since the LOMAC LKM is apt to be loaded relatively late in the
 * boot, many processes are apt to be running when LOMAC is first
 * initialized.  We must give each of these pre-existing processes its
 * attrs.  The present strategy is to "grandfather in" these processes
 * at the highest possible level.  Furthermore, these grandfathered
 * processes are apt to already have a number of files open at LOMAC
 * initialization time.  We must also set the lattrs of all of these
 * files.
 *
 */

void handle_grandfathering( void ) {

  struct task_struct *p_proc;           /* traverses the process table */
  unsigned long fd;    /* traverses `p_proc's list of file descriptors */
  struct file *p_file;        /* points to file structs for open files */

  /* For each existing process, set its level, and the levels of *
   * all of the files it currently has open.                     */
  p_proc = current;
  do {

    /* Set LOMAC attributes of the process indicated by `p_proc'. */
    set_subject_lattr( p_proc, LATTR_HIGH );

    /* Now set the levels of all of the files `p_proc' has open. */
    if( p_proc->files ) {   /* some kernel procs don't have `files' */
      for( fd = 0; fd < p_proc->files->max_fds; fd++ ) {
	if( ( p_file = p_proc->files->fd[ fd ] ) && ( p_file->f_dentry ) ) {

	  if( IS_LOCAL_SOCKET( p_file->f_dentry ) ) {

	    grandfather_unix_socket( p_proc, p_file->f_dentry );

	  } else if( IS_PIPE( p_file->f_dentry ) ) {

	    grandfather_pipe( p_proc, p_file->f_dentry );

	  } else {

	    grandfather_named_object( p_proc, p_file->f_dentry );
	  
	  } /* if/elsif/else on file type */

	} /* if fd corresponds to an open file */

      } /* for all fd's in `p_proc's list */

    } /* if this is a proc that has files */

  } while( ( p_proc = p_proc->next_task ) != current );  /* for each proc */

} /* handle_grandfathering() */


/* grandfather_pipe()
 *
 * in:     p_proc - process that has `p_pipe' open
 *         p_pipe - dentry of an unnamed pipe 
 * out:    p_pipe - will have its level set
 * return: nothing
 *
 * This function sets the initial level of an unnamed pipe that was created
 * before LOMAC load-time.
 *
 */

void grandfather_pipe( struct task_struct *p_proc, struct dentry *p_pipe ) {

  /* It's safe to start unnamed pipes at the highest level. *
   * They will be demoted properly the first time someone   *
   * writes to them.                                        */
  set_object_lattr( p_pipe, LATTR_HIGH );
  
  if( verbose & VERBOSE_PIPE ) {
    log_start();
    log_append_string( "LOMAC/K: " );
    log_append_subject_id( p_proc );
    log_append_string( " grandfathered with unnamed pipe " );
    log_append_object_id( p_pipe );
    log_append_string( " level " );
    log_append_int( LOMAC_HIGHEST_LEVEL );
    log_append_string( ".\n" );
    log_print();
  }

} /* grandfather_pipe() */


/* grandfather_named_object()
 *
 * in:     p_proc   - process that has `p_object' open
 *         p_object - dentry of a named object (usually file)
 * out:    p_object - will have its level set
 * return: nothing
 *
 * This function sets the initial level of a named object (such as a
 * file) that was opened before LOMAC load-time.
 *
 * This function does nothing and returns quietly if `p_object' turns
 * out not to have a name in the filesystem.  This is an omission to
 * save time - we should probably fix the code so this function is
 * not called in those cases.
 *
 */

void grandfather_named_object( struct task_struct *p_proc,
			       struct dentry *p_object ) {

  lattr_t object_lattr;      /* holds LOMAC attributes for open files */

  if( !get_object_canabspath( p_object, log_s, LOG_BUFFER_LENGTH ) ) {
	    
    get_pathname_lattr( log_s, &object_lattr );
    set_object_lattr( p_object, object_lattr );
	    
    if( verbose & VERBOSE_LOG_OBJECTS ) {
      log_start();
      log_append_string( "LOMAC/K: " );
      log_append_subject_id( p_proc );
      log_append_string( " grandfathered with object " );
      log_append_object_id( p_object );
      log_append_string( " level " );
      log_append_int( object_lattr.level );
      log_append_string( "\n" );
      log_print();
    }
    
  } /* if/else file has pathname */

} /* grandfather_named_object() */


/* grandfather_unix_socket()
 *
 * in:     p_proc   - process that has `p_socket' open
 *         p_socket - a UNIX (local) domain socket that was opened before
 *                    LOMAC load time
 * out:    p_socket - will have its level set.  If `p_socket' is bound to
 *                    a named object in the filesystem, the named object
 *                    will have its level set, too.
 * return: nothing
 *
 * This function sets the levels of UNIX (local) domain sockets that
 * were opened before LOMAC load time.  It also sets the levels of
 * the named objects the sockets are bound to (if any).
 *
 * This code is a temporary, simple solution that provides safety
 * (that is, it prevents the use of pre-existing UNIX domain sockets
 * for polluting high-level processes) but makes some sockets low that
 * ought to be high.
 *
 * If `p_socket' is bound to a named object in the filesystem, then 
 * this function will set that named object's level correctly, and
 * it will set `p_socket' to the same level.  This much is proper.
 * However, in all other cases, this function sets `p_socket's level
 * to LOMAC_LOWEST_LEVEL.  This is simple and safe, but sometimes
 * these sockets deserve to be at a higher level.  This will have
 * to be fixed eventually.  Since the few sockets that exist at
 * LOMAC load time on my test system seem to deserve to be low,
 * I'll proceed with this code for now.
 */

void grandfather_unix_socket( struct task_struct *p_proc,
			      struct dentry *p_socket ) {

  lattr_t lattr;                 /* proper LOMAC attributes for `p_socket'. */
  struct dentry *p_named_object;  /* name of bound sockets, like "/dev/log" */

  /* The default level for a pre-existing socket is low, (for now). */
  lattr = LATTR_LOW;

  /* If `p_socket' is bound to a named object, then set the named  *
   * object's level, and then save that level in `level' so we can *
   * set `p_socket' that way, later.                               */
  if( ( p_named_object = socket_name( p_socket ) ) ) {
  
    grandfather_named_object( p_proc, p_named_object );
    get_object_lattr( p_named_object, &lattr );

#ifdef PARANOID
    if( lattr.level == LOMAC_INVALID_LEVEL ) {
      PANIC( "LOMAC/K: invalid named object level for bound socket.\n" );
    }
#endif

  } /* if socket is bound to named object */

  /* set `p_socket's LOMAC attributes and log the event. */
  set_object_lattr( p_socket, lattr );
  
  if( verbose & VERBOSE_LOG_SOCKETS ) {
    log_start();
    log_append_string( "LOMAC/K: " );
    log_append_subject_id( p_proc );
    log_append_string( " grandfathered with local socket " );
    log_append_object_id( p_socket );
    log_append_string( " level " );
    log_append_int( lattr.level );
    log_append_string( ".\n" );
    log_print();
  }

} /* grandfather_unix_socket() */
