/***************************************************************************
 *
 * kernel_util.c
 *
 * LOMAC - Low Water-Mark Mandatory Access Control for Linux
 * Copyright (c) 1999, 2000, 2001, 2002 Networks Associates
 * Technology, Inc.  All rights reserved.
 * 
 * This file is part of LOMAC.
 *
 * LOMAC is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * LOMAC is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with LOMAC; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *
 * This file contains some kernel-specific utility functions used by 
 * routines in other kernel_*.c files.  These utility functions are
 * kernel-specific - they should not contains any lomac-specific
 * types or function calls.  These functions are not visible to the
 * code in the lomac_*.c files.
 *
 * Some of these functions don't do proper locking.  This isn't a
 * problem on single-CPU systems, but it's ugly.  Thesse functions
 * should eventually be replaced or fixed.  ##
 *
 ***************************************************************************/

#include "kernel_util.h"
#include "kernel_interface.h"
#include "lomac_plm.h"

#include <linux/file.h>    /* for fcheck() */
#include <linux/net.h>     /* to support net/sock.h */
#include <net/sock.h>      /* for struct sock, struct unix_opt */
#include <asm/uaccess.h>   /* for copy_from_user() */


int remove_last_component( char *path_s, int null_index );

/* --------------------- exported functions ------------------------ */


/* kernel_namei()
 *
 * in:     path_s - path to lookup
 * out:    nothing
 * return: as the kernel's namei() function.  Use IS_ERR() and ERR_PTR().
 *
 *     This function provides an alternative to the kernel's namei()
 * which (unlike the kernel version) takes its path argument from 
 * kernel-space.
 * 
 *     When this function returns a valid dentry, it will increment
 * the dentry's reference count.  The caller must eventually call
 * pdut() on that dentry.
 *
 */

struct dentry *kernel_namei( const char *path_s ) {

  struct dentry *p_dentry;   /* points to `path_s's dir cache entry. */

  p_dentry = lookup_dentry( path_s, NULL, 1 );  /* 1 means follow links */
  if( !IS_ERR( p_dentry ) ) {
    if( ! ( p_dentry->d_inode ) ) {
      dput( p_dentry );
      return( ERR_PTR( -ENOENT ) );
    }
  }
  return( p_dentry );

} /* kernel_namei() */



/* pid_to_task()
 *
 * in:     pid - pid of the task to find
 * out:    nothing
 * return: pointer to task struct corresponding to `pid' or NULL if not found
 *
 * This is a module-friendly replacement of the find_task_by_pid macro
 * in linux/sched.h .
 *
 */

struct task_struct *pid_to_task( int pid ) {

  struct task_struct *p = current;

  do {
    if( p->pid == pid ) {
      return( p );
    }
    p = p->next_task;
  } while( p != current );

  return( NULL );

} /* pid_to_task() */


/* pgrp_to_task()
 *
 * in:     pgrp - pgrp of the task to find
 * out:    nothing
 * return: pointer to task struct corresponding to `pgrp' or NULL if not found
 *
 * Similar to pid_to_task(), but finds first occurrence of task with the
 * specified `pgrp' instead of pid.
 *
 */

struct task_struct *pgrp_to_task( int pgrp ) {

  struct task_struct *p = current;

  do {
    if( p->pgrp == pgrp ) {
      return( p );
    }
    p = p->next_task;
  } while( p != current );

  return( NULL );

} /* pgrp_to_task() */
    


/* socket_peer()
 *
 * in:     p_socket - one of a pair of connected UNIX (local) domain sockets.
 * out:    nothing
 * return: the other socket in the pair, or NULL if it cannot be found.
 *
 * Given one UNIX (local) domain socket in a connected pair, this function
 * will find the other socket.  Note that the sockets are represented
 * by their dentries.
 *
 */

struct dentry *socket_peer( struct dentry *p_socket_dentry ) {

  struct socket *p_socket;        /* points to `p_socket_dentry's socket */
  struct sock   *p_sock;          /* points to `p_socket's sock struct   */

#ifdef PARANOID
  if( !p_socket_dentry ) {
    panic( "LOMAC: socket_peer called on null socket.\n" );
  }
  if( !( IS_LOCAL_SOCKET( p_socket_dentry ) ) ) {
    panic( "LOMAC: socket_peer called on non-UNIX-domain socket.\n" );
  }
#endif

  if( !( p_socket_dentry->d_inode ) ) {
    return( NULL );
  }
  p_socket = &(p_socket_dentry->d_inode->u.socket_i);

  if( !p_socket ) {
    return( NULL );
  }
  p_sock = (struct sock *)p_socket->sk;

  if( !( p_sock ) ) {
    return( NULL );
  }
  if( !( p_sock->pair ) ) {
    return( NULL );
  }
  return( p_sock->pair->protinfo.af_unix.dentry );

} /* socket_peer() */


/* socket_name()
 *
 * in:     p_socket - a UNIX (local) domain socket that is bound to a name.
 * out:    nothing
 * return: the dentry of the name of the socket in the filesystem, or
 *         NULL if this cannot be determined.
 *
 * Given a UNIX (local) domain socket that is bound to a name in the
 * filesystem, this function will find the dentry corresponding to the
 * name.
 *
 */

struct dentry *socket_name( struct dentry *p_socket_dentry ) {

  struct socket *p_socket;        /* points to `p_socket_dentry's socket */
  struct sock   *p_sock;          /* points to `p_socket's sock struct   */

#ifdef PARANOID
  if( !p_socket_dentry ) {
    panic( "LOMAC: socket_peer called on null socket.\n" );
  }
  if( !( IS_LOCAL_SOCKET( p_socket_dentry ) ) ) {
    panic( "LOMAC: socket_peer called on non-UNIX-domain socket.\n" );
  }
#endif

  if( !( p_socket_dentry->d_inode ) ) {
    return( NULL );
  }
  p_socket = &(p_socket_dentry->d_inode->u.socket_i);

  if( !p_socket ) {
    return( NULL );
  }
  p_sock = (struct sock *)p_socket->sk;

  if( !( p_sock ) ) {
    return( NULL );
  }
  return( p_sock->protinfo.af_unix.dentry );

} /* socket_name() */


/* copy_sockaddr_to_kernel()
 *
 * in:     p_user_sockaddr   - pointer to sockaddr in user-space
 *         length            - length of sockaddr
 * out:    p_kernel_sockaddr - `*p_user_sockaddr' copied here
 * return: value      condition
 *         -----      ---------
 *           0        success
 *         -EINVAL    `length' > MAX_SOCK_ADDR
 *         -EFAULT    copyin failed
 *
 * This function is a replacement for move_addr_to_kernel() from
 * linux/net/socket.c - a useful function that is unfortunately
 * not exported to modules.
 */

int copy_sockaddr_to_kernel( void *p_user_sockaddr, int length, 
			     void *p_kernel_sockaddr ) {

  int ret_val;       /* value returned to caller */

  /* if there's nothing to copy, don't bother copying */
  if( length != 0 ) {
    
    ret_val = -EINVAL;
    if( length > MAX_SOCK_ADDR ) {
      goto out;
    }
    if( length < 0 ) {
      goto out;
    }
    
    ret_val = -EFAULT;
    if( copy_from_user( p_kernel_sockaddr, p_user_sockaddr, length ) ) {
      goto out;
    }

  } /* if non-zero `length' */

  ret_val = 0;   /* success */

 out:

  return( ret_val );

} /* copy_sockaddr_to_kernel() */



/* make_canabspath()
 *
 * in:     path_s       - path to object that may or may not yet exist;
 *                        path may or may not be canonical or absolute;
 *                        path may or may not be valid.  Path must be
 *                        in kernel-space.
 *         canabspath_s - pointer to a kernel page to hold output canabspath.
 *         pp_dir_dentry- pointer to a dentry pointer for the directory that
 *                        would be modified by the creation of the file
 *                        named by `path_s'.
 *         pp_dentry    - pointer to a dentry pointer for `path_s's dentry.
 * out:    canabspath_s - On success, this buffer will receive the absolute
 *                        path in canonical form corresponding to `path_s'.
 *                        On failure, this buffer will receive garbage.
 *                        This function does not do checks to avoid exceeding
 *                        the length of this buffer on the expectation that
 *                        the buffer will always be an entire page in size -
 *                        longer than any path string.
 *         pp_dir_dentry- `*pp_dir_dentry' will be set to point to the dentry
 *                        of the leafmost directory that would be modified
 *                        by the creation of the file named by `path_s'.  If
 *                        this directory does not exist, `*pp_dir_dentry'
 *                        will be cleared to NULL and the function will
 *                        return an error code.
 *         pp_dentry    - if `canabspath_s' exists, `*pp_dentry' will be set
 *                        to its dentry.  The caller must dput( *pp_dentry).
 *                        On the other hand, if `canabspath_s' does not
 *                        exist, `*pp_dentry' will be set to NULL.
 * return: 0 on success, else kernel namei()'s error codes.
 *
 * This function is used to predict the absolute, canonical-form
 * pathnames of objects that are about to be opened, based on the
 * not-necessarily absolute, canonical-form, or valid paths supplied
 * by the calling process.  Due to the nature of the operating system,
 * given a particular `path_s', there are three cases:
 *
 * 1)  `path_s' describes an existing object in the file system.
 *     In this case, we convert `path_s' into absolute, canonical form,
 *     and output this new canonical-form path out in `canabspath_s'.
 * 2)  `path_s' does not describe an existing object in the file system,
 *     but `path_s' with its last component removed describes a directory.
 *     This case may occur when the calling process wants to create a
 *     new filesystem object in that directory.  In this case, we
 *     convert the directory's path into absolute canonical form, append
 *     the last component of `path_s', and output this in `canabspath_s'.
 * 3)  `path_s' does not describe any object in the filesystem, and when
 *     its last component is removed, it still does not describe a
 *     directory in the filesystem.  In this case, we return an error
 *     code, following the normal error behavior of the open (etc.) 
 *     system calls.
 *
 * `path_s' describes a file that may or may not exist.  If this function
 * finds that `path_s' names an already-existing file, it will set 
 * `*pp_dentry' to the dentry for that file.  Otherwise it will clear
 * `*pp_dentry' to NULL.  Callers may examine `*pp_dentry' to determine
 * whether or not `path_s' names an existing file, and if it does, what
 * kind of file (file, pipe, socket, etc.) it is.  Note that the caller
 * must dput( *pp_dentry ) if `*pp_dentry' is set by this function.
 *
 * `path_s' describes a pathname, possibly not in canonical absolute
 * form.  The last component in the path may or may not exist.  The
 * second-to-last component in the path must be an existing directory,
 * or this function will return an error value.  For example, if `path_s'
 * is "foo/bar", then "bar" may or may not name an existing file, but 
 * "foo" must name an existing directory.  This requirement is due to
 * the fact that the kernel won't let a process create more than one
 * directory with a single call - the kernel doesn't do "mkdir -p".
 * This function will set `*pp_dir_dentry' to the dentry of this
 * second-to-last-component directory if it exists; otherwise (on error),
 * it will clear it to null.  Callers may use `*pp_dir_dentry' to
 * identify the directory that will be modified by the creation or
 * removal of the file named by the last component of `path_s'. Note that
 * the caller must eventually call dput( *pp_dir_dentry ) if this function
 * sets `*pp_dir_dentry'.
 *
 * This function will also set the LOMAC attribtues of `*pp_dir_dentry'.
 *
 */

int make_canabspath( const char *path_s, char *canabspath_s, 
		     struct dentry **pp_dir_dentry,
		     struct dentry **pp_dentry ) {

  lattr_t dir_lattr;         /* LOMAC attributes of parent dir */
  int path_len;              /* string length of `path_s' */
  int last_component_index;  /* index of last component in `path_s' */
  /* example:  for path_s = "/a//b///", last_component_index = 4.   */
  int ret_val;               /* holds error return codes */

#ifdef PARANOID
  if( !path_s ) {
    panic( "LOMAC/K: make_canabspath() given null path_s.\n" );
  }
  if( !canabspath_s ) {
    panic( "LOMAC/K: make_canabspath() given null canabspath_s.\n" );
  }
  if( !pp_dentry ) {
    panic( "LOMAC/K: make_canabspath() given null dentry.\n" );
  }
  if( !pp_dir_dentry ) {
    panic( "LOMAC/K: make_canabspath() given null dir dentry.\n" );
  }
#endif

  /* Copy `path_s' into `canabspath_s' where we can examine and modify *
   * it.  try to get a dentry for the full path, then back up one      *
   * component, and try to get a dentry for the parent directory.      */
  strncpy( canabspath_s, path_s, PATH_MAX );
  path_len = strnlen( canabspath_s, PATH_MAX );
  *pp_dentry = kernel_namei( canabspath_s );
  last_component_index = remove_last_component( canabspath_s, path_len );
  *pp_dir_dentry = kernel_namei( canabspath_s );

  /* Now that we have all the information we need, we can check for *
   * some easily-handled error conditions.  If we couldn't get a    *
   * dentry for the parent dir, or if we did get a dentry but it    *
   * isn't a directory, then we have a case-3 situation.            */
  ret_val = PTR_ERR( *pp_dir_dentry );
  if( IS_ERR( *pp_dir_dentry ) ) {

    /* Both the dentry pointers should be cleared on failure.   *
     * We can clear `*pp_dentry' without dput()'ing it, because *
     * if `*pp_dir_dentry' is bad, `*p_dentry' must be too      */
    *pp_dir_dentry = NULL;
    *pp_dentry = NULL;  
    return( ret_val );

  } /* if failed to get dentry for parent directory */

  if( !( S_ISDIR( (*pp_dir_dentry)->d_inode->i_mode ) ) ) {

    /* Not a directory, this is a malformed path.  Null the dentry *
     * pointers.  Again, we can assume that `*p_dentry' is bad and *
     * needs no dput() since `*pp_dir_dentry' is not a directory.  */
    dput( *pp_dir_dentry );
    *pp_dir_dentry = NULL;
    *pp_dentry = NULL;
    return( -ENOTDIR );

  } /* if `*pp_dir_dentry' is not a dir */

  /* Put the canabspath of the parent dir in `canabspath_s' and use it *
   * to set the lattr of *pp_dir_dentry.  Leave the canabspath in      *
   * `canabspath_s', we'll use it later if we have a case 2 situation. */
  strncpy( canabspath_s, d_path( *pp_dir_dentry, canabspath_s, PATH_MAX ),
	   PATH_MAX ); 
  get_pathname_lattr( canabspath_s, &dir_lattr );
  set_object_lattr( *pp_dir_dentry, dir_lattr );

  /* If we failed to get `*pp_dentry' then the file named by `path_s' *
   * does not exist or is unobtainable due to -EACCES.  Clear         *
   * `*pp_dentry' to indicate this condition  to the caller.          */
  ret_val = PTR_ERR( *pp_dentry );
  if( IS_ERR( *pp_dentry ) ) {

    *pp_dentry = NULL;
    if( ret_val == -EACCES ) {
      /* If we failed to get `*pp_dentry' due to -EACCES, then the file   *
       * named by `path_s' exists, but we can't touch it.  Any operations *
       * on it will fail with -EACCES, so we'll just return -EACCES now.  */
      dput( *pp_dir_dentry );
      *pp_dir_dentry = NULL;
      return( ret_val );

    } /* if access denied */
  } /* if no `*pp_dentry' */


  /* Now we can try to handle cases.
   * Case 1: `path_s' refers to a valid filesystem object.             *
   * Case 2: `path_s' is a well-formed path: although it does not name *
   *          an existing filesystem object, the kernel could create   *
   *          the named filesystem object if asked to do so.           */

  if( *pp_dentry ) {

    /* This is a case 1 situation.                                      *
     * `canabspath_s' does indeed refer to a valid object, but it may   *
     * not be in canonical absolute form.  Get the canonical absolute   *
     * form of the path from `*pp_dentry', copy it into `canabspath_s', *
     * and return.                                                      */
    strncpy( canabspath_s, 
	     d_path( *pp_dentry, canabspath_s, PATH_MAX ),
	     PATH_MAX ); 

  } else {

    /* This is a case 2 situation.                                     *
     * `canabspath_s' does not describe an existing filesystem object, *
     * but `canabspath_s' with its final component removed does name   *
     * an existing directory and we can construct a canonical absolute *
     * path by converting it into absolute, canonical form, append the *
     * last component of `path_s' to it.                               */

    /* The canabspath for `p_dir_dentry' should still be in canabspath_s. */

    /* Put a "/" between `*pp_dir_dentry's path and the last component *
     * from path_s that we'll be adding in the next step.              *
     * Exception: we don't need to do this if the path in              *
     * `canabspath_s' is the root directory ("/").                     */
    if( !( canabspath_s[ 0 ] == '/' && canabspath_s[ 1 ] == '\0' ) ) {
      strncat( canabspath_s, "/", 2 );
    }
 
    /* Append last component of `path_s'. */
    strncat( canabspath_s, &(path_s[ last_component_index ]),
	     (path_len - last_component_index ) );

  } /* case 1 vs. case 2 situation */

  return( 0 );

} /* make_canabspath() */


/* ---------------------- local functions --------------------------- */


/* remove_last_component()
 *
 * in:     path_s     - pathname
 *         null_index - length of string in `path_s'
 * out:    path_s     - pathname with last component removed
 * return: the index of the first character in the last component
 *         of the original, unmodified `path_s'.
 * 
 * this function modifies `path_s', removing its last component.  It
 * returns the index of the first character of the last component as
 * follows:
 * 
 *  input path_s = "a/b//cd///"
 *        index:    0123456789
 * output path_s = "a/b"
 *       return:         5
 *
 */
 
int remove_last_component( char *path_s, int null_index ) {

  int last_component_index;      /* index of last component for return */

  /* First, remove any trailing slashes.  As a special case, we don't  *
   * want to remove a slash at index 0, since this represents the root *
   * directory.                                                        */
  for( null_index--; /* start with char to the left of null terminator */
       ( null_index > 0 && path_s[ null_index ] == '/' ); 
       null_index-- ) {
    path_s[ null_index ] = '\0';
  }

  /* Next, remove any non-"/" chars until we find another "/".  It's OK   *
   * to remove every remaining character in the string - and empty        *
   * string represents the `current' process's current working directory. */
  last_component_index = null_index;       /* first guess at proper index */
  for( ;               /* start with the same char previous loop ended on */
       ( null_index >= 0 && path_s[ null_index ] != '/' ); 
       null_index-- ) {
    path_s[ null_index ] = '\0';
    last_component_index = null_index;    /* update guess at proper index */
  }

  /* Finally, remove any slashes that separated the last component     *
   * from the second-to-last component.  As a special case, we don't   *
   * want to remove a slash at index 0, since this represents the root *
   * directory.                                                        */
  for( ;            /* start with the same char previous loop ended on */
       ( null_index > 0 && path_s[ null_index ] == '/' ); 
       null_index-- ) {
    path_s[ null_index ] = '\0';
  }

  return( last_component_index );

} /* remove_last_component() */




