#ifndef _KERNEL_UTIL_H_
#define _KERNEL_UTIL_H_

/***************************************************************************
 *
 * kernel_util.h
 *
 * LOMAC - Low Water-Mark Mandatory Access Control for Linux
 * Copyright (c) 1999, 2000, 2001, 2002 Networks Associates
 * Technology, Inc.  All rights reserved.
 * 
 * This file is part of LOMAC.
 *
 * LOMAC is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * LOMAC is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with LOMAC; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *
 * This is the public interface to kernel_util.c.  kernel_util.c
 * contains architecture-dependent utility functions that are
 * not visible to the architecture-independent core of LOMAC.
 *
 *
 ***************************************************************************/

#include <asm/uaccess.h>   /* for KERNEL_DS and USER_DS */
#include <linux/sched.h>   /* for `current' process, struct task */
#include <linux/fs.h>      /* for inode, dentry, file, iovec structures */

/* In BSD, one of the arguments to namei allows the caller to indicate *
 * whether the filename lies in kernel space, or needs to be copied    *
 * into kernel space from user space.  Linux does not have this        *
 * argument.  Instead, it provides a mechanism for turning the         *
 * kernel-space vs. user-space pointer argument checks on and off for  *
 * a given process.  The following macros provide a meaningfully-named *
 * interface to this mechanism, so LOMAC can call namei with filename  *
 * arguments in kernel-space.                                          */

#define TURN_ARG_CHECKS_OFF  set_fs(KERNEL_DS)
#define TURN_ARG_CHECKS_ON   set_fs(USER_DS)


/* Here's a macro to identify objects that are (unnamed) pipes.        *
 * In the 2.0 kernel, we could check to see if `p_lvnode->i_pipe' is   *
 * set.  Unfortunately, the 2.2 kernel's get_pipe_inode() function     *
 * forgets to set this flag.  So, we identify pipes/fifos with the     *
 * S_IFIFO flag, and distinguish between the two by their i_op fields. */
#define IS_PIPE( p_pipe ) \
     ( (p_pipe) && \
       (p_pipe)->d_inode && \
       (p_pipe)->d_inode->i_mode & S_IFIFO && \
       (p_pipe)->d_inode->i_op != &fifo_inode_operations )

/* Here's macros to identify objects which are local and non-local *
 * sockets.                                                        */
#define IS_SOCKET( p_socket ) \
     ( (p_socket) && \
     (p_socket)->d_inode && \
     (p_socket)->d_inode->i_sock )

#define IS_LOCAL_SOCKET( p_socket ) \
     ( (p_socket) && \
     (p_socket)->d_inode && \
     (p_socket)->d_inode->i_sock && \
     (p_socket)->d_inode->u.socket_i.ops->family == PF_UNIX )

#define IS_NETWORK_SOCKET( p_socket ) \
     ( (p_socket) && \
     (p_socket)->d_inode && \
     (p_socket)->d_inode->i_sock && \
     (p_socket)->d_inode->u.socket_i.ops->family != PF_UNIX )


#define MAX_SOCK_ADDR 128 /* max length of sockaddr from linux/net/socket.c */

struct dentry *kernel_namei( const char *path_s );
struct task_struct *pid_to_task( int pid ); 
struct task_struct *pgrp_to_task( int pgrp ); 
struct dentry *socket_peer( struct dentry *p_socket_dentry );
struct dentry *socket_name( struct dentry *p_socket_dentry );
int copy_sockaddr_to_kernel( void *p_user_sockaddr, int length, 
			     void *p_kernel_sockaddr );

int make_canabspath( const char *path_s, char *canabspath_s, 
		     struct dentry **pp_dir_dentry,
		     struct dentry **pp_dentry );
#endif
