/***************************************************************************
 *
 * kernel_wrappers.c
 *
 * LOMAC - Low Water-Mark Mandatory Access Control for Linux
 * Copyright (c) 1999, 2000, 2001, 2002 Networks Associates
 * Technology, Inc.  All rights reserved.
 * 
 * This file is part of LOMAC.
 *
 * LOMAC is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * LOMAC is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with LOMAC; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *
 * This file contains functions which "wrap" kernel system calls, and
 * a number of utility functions to serve them.  When the LKM loads,
 * the initialization routines in kernel_lkm.c modify the system call
 * table jump table by replacing entries for system calls that are
 * interesting to the LOMAC LKM with entries for corresponding
 * "wrapper" functions in this file.  When a user-space program calls
 * one of these interesting system calls, the replacement wrapper
 * function gets called instead.
 *
 * The wrapper functions all have a similar structure.  They all take
 * exactly the same arguments as the system call functions they
 * replace.  Their first step is to allow the LOMAC LKM to examine
 * these parameters in order to do monitoring or mediation.  Second,
 * (if the mediation result allows it) they call the original system
 * call, giving the calling user-space program the service it expects.
 * Finally, they allow the LOMAC LKM a second chance to do monitoring
 * after the original system call has returned.
 *
 * There are some serious time-of-check/time-of-use bugs in these
 * functions that need to be addressed.
 *
 ***************************************************************************/

#include <linux/sched.h>   /* for `current' process */
#include <linux/fs.h>      /* for inode, dentry, file, iovec structures */
#include <linux/file.h>    /* for fcheck() */
#include <linux/net.h>     /* for SYS_BIND and friends... */
#include <linux/socket.h>  /* for struct sockaddr */
#include <linux/un.h>      /* for struct sockaddr_un */
#include <net/sock.h>      /* for struct sock, struct unix_opt */
#include <asm/uaccess.h>   /* for copy_from_user() */

#include "kernel_wrappers.h"
#include "kernel_util.h"
#include "lomac_monitor.h"
#include "lomac_mediate.h"
#include "lomac_log.h"


/* LOMAC always allows writes to character devices such as serial  *
 * ports, ttys, pseudo-ttys, and /dev/null.  This macro identifies *
 * such devices, for the convenience of the creat/write mediation  *
 * handling functions.                                             */

#define WRITE_EXEMPT( pd ) \
  ( ( (pd)->d_inode->i_mode & S_IFCHR ) &&     /* if char device */     \
    ( ( MAJOR( (pd)->d_inode->i_rdev ) == 5 )   ||     /* cua  */       \
      ( MAJOR( (pd)->d_inode->i_rdev ) == 4 )   ||     /* tty  */       \
      ( MAJOR( (pd)->d_inode->i_rdev ) == 3 )   ||     /* ttyp */       \
      ( MAJOR( (pd)->d_inode->i_rdev ) == 136 ) ||     /* pts  */       \
      ( MAJOR( (pd)->d_inode->i_rdev ) == 2 )   ||     /* pty  */       \
      ( MAJOR( (pd)->d_inode->i_rdev ) == 6 )   ||     /* lp  */        \
      ( ( MAJOR( (pd)->d_inode->i_rdev ) == 10 ) &&    /* /dev/psaux */ \
        ( MINOR( (pd)->d_inode->i_rdev ) == 1 ) ) ||                    \
      ( ( MAJOR( (pd)->d_inode->i_rdev ) == 1 ) &&     /* /dev/null */  \
        ( MINOR( (pd)->d_inode->i_rdev ) == 3 ) ) ) ) 

#ifdef PARANOID

/* Each wrapper function sets this pointer to point at a string *
 * corresponding to the name of the kernel function it wraps.   *
 * The PANIC function prints this name on panic, so we get a    *
 * useful error message like "panic while handling open".       */
char *global_wrapper_name_s = "none";  
#endif


void handle_monitor_read( struct dentry *p_dentry );
int handle_mediate_write( const char *op_s, struct dentry *p_dentry );
int handle_accept( int call, unsigned long *args );
int handle_bind( int call, unsigned long *args );
int handle_connect( int call, unsigned long *args );
int handle_recv( int call, unsigned long *args );
int handle_socketpair( int call, unsigned long *args );

/**************************************************************************
 *
 * wrapper functions (functions that intercept system calls and call
 * the higher-level LOMAC routines to react to them)...
 *
 **************************************************************************/


/* wrap_open()
 *
 * Processes can use the open system call to create new files, or to
 * open existing files.  In both cases, we must call `monitor_open()'
 * to prompt LOMAC to assign a level to the inode resulting from the
 * open (assuming the open succeeded).  When a process uses the open
 * system call to create a new file, we must also call
 * `monitor_creat()' to allow LOMAC to mediate (allow/deny) the
 * create.
 *
 * LOMAC attempts to provide the following guarantees:
 * o No process will create a file whose level is higher than its own.
 *   Exception: creats are always allowed on files like serial devices
 *   and pseudo-terminals.  See WRITE_EXEMPT macro for complete list of
 *   exempt files.
 * o No process will truncate a file whose level is higher than its own.
 *   Exception: truncates are always allowed on files like serial devices
 *   and pseudo-terminals.  See WRITE_EXEMPT macro for complete list of
 *   exempt files.
 * o No process can create a file in a directory whose level is higher
 *   than its own.
 *
 */

int wrap_open( const char *filename, int flags, int mode ) {

  char *k_filename_s;          /* kernel-space copy of `filename' */
  char *k_canabspath_s;        /* kernel-space canabs copy of `filename' */
  struct dentry *p_dentry;     /* dentry for `filename' if one exists */
  struct dentry *p_dir_dentry; /* dentry for `filename's parent directory */
  struct file *p_file;         /* file struct generated by successful open */
  int ret_val;                 /* holds return values */

#ifdef PARANOID
  global_wrapper_name_s = "open";
#endif

  /* make `k_canabspath_s' a kernel-space copy of `filename' in canonical *
   * absolute form.  Make `p_dentry' point to the dentry for `filename'   *
   * if `filename' describes an existing file.  Make `p_dir_dentry' point *
   * to its parent directory.                                             */
  if( IS_ERR( ( k_filename_s = getname( filename ) ) ) ) {
    return( PTR_ERR( k_filename_s ) );
  }
  if( !( k_canabspath_s = (char *)__get_free_page( GFP_KERNEL ) ) ) {
    ret_val = -ENOMEM;
    goto out_putname;
  }
  if( ( ret_val = make_canabspath( k_filename_s, k_canabspath_s,
				   &p_dir_dentry, &p_dentry ) ) ) {
    goto out_dputs;
  }

  /* If `flags' has the O_TRUNC flag set, then the `current' process  *
   * is trying to create a zero-length file (truncate) named by       *
   * `filename'.  This action is effectively a write, so we must      *
   * perform mediation before calling the kernel's open system call.  *
   * We also wish to prevent the `current' process from creating a    *
   * file with a level higher than its own.  So, we will also perform *
   * mediation if `flags' contains O_CREAT and `filename' does not    *
   * name an already-existing file (according to p_dentry).           */

  /* If we should mediate ... */
  if( ( flags & O_TRUNC ) || 
      ( ( flags & O_CREAT ) && ( !p_dentry ) ) ) {

    /* If this isn't an exempt case ... */
    if( ! ( p_dentry && WRITE_EXEMPT( p_dentry ) ) ) { 
    
      /* mediate */
      if( !( mediate_subject_path( "open", current, k_canabspath_s ) ) ) {
	ret_val = -EACCES;
	goto out_dputs;
      }
      if( !( mediate_subject_object( "open", current, p_dir_dentry ) ) ) {
	ret_val = -EACCES;
	goto out_dputs;
      }

    } /* if this is not an exempt case */

  } /* if we should mediate */

  /* Make open system call using our k_canabspath_s argument. */ 
  TURN_ARG_CHECKS_OFF;
  ret_val = ((int (*)(const char *, int, int))orig_open)
    ( k_canabspath_s, flags, mode );
  TURN_ARG_CHECKS_ON;

  /* If the open system call succeeded, prompt LOMAC to set the level *
   * of the inode it created to represent the opened file.            */
  if( ret_val >= 0 ) {
    p_file = fget( ret_val );
#ifdef PARANOID
    if( !p_file ) {
      /* fget shouldn't fail here, since the open just succeeded. */
      PANIC( "LOMAC/K: unexpected fget failure in wrap_open.\n" );
    }
#endif
    monitor_open( current, p_file->f_dentry );
    fput( p_file );
  }

 out_dputs:
  if( p_dir_dentry ) {
    dput( p_dir_dentry );
  }
  if( p_dentry ) {
    dput( p_dentry );
  }
  free_page( (unsigned long)k_canabspath_s );
 out_putname:
  putname( k_filename_s );
  return( ret_val );

} /* wrap_open() */ 


/* wrap_read()
 *
 * LOMAC attempts to provide the following guarantees:
 * o Upon reading from a file, local-domain socket, network
 *   socket, unnamed pipe, or a FIFO with a level lower than its own,
 *   the level of the current process's job will be reduced to match
 *   the object's level.
 *
 */

int wrap_read( unsigned int fd, char *buf, int count ) {

  int ret_val;               /* holds return value of original syscall */
  struct file *p_file;       /* file structure corresponding to `fd' */

#ifdef PARANOID
  global_wrapper_name_s = "read";
#endif

  ret_val = ((int (*)(unsigned int, char *, int))orig_read)( fd, buf, count );

  if( ret_val >= 0 ) {  /* If read succeeded, we must monitor it. */

    p_file = fget( fd );
#ifdef PARANOID
    /* fget should not fail since read just succeeded. */
    if( !p_file ) {
      PANIC( "LOMAC/K: unexpected fget failure in wrap_read.\n" );
    }
#endif
    handle_monitor_read( p_file->f_dentry );
    fput( p_file );

  } /* if read succeeded */

  return( ret_val );

} /* wrap_read() */


/* wrap_readv()
 *
 */

int wrap_readv( unsigned long fd, const struct iovec *vector, long count ) {

  int ret_val;               /* holds return value of original syscall */
  struct file *p_file;       /* file structure corresponding to `fd' */

#ifdef PARANOID
  global_wrapper_name_s = "readv";
#endif

  ret_val = ((int (*)(unsigned long, const struct iovec *, long))orig_readv)
    ( fd, vector, count );

  if( ret_val >= 0 ) {  /* If readv succeeded, we must monitor it. */

    p_file = fget( fd );
#ifdef PARANOID
    if( !p_file ) {
      /* fget should not fail since readv just succeeded. */
      PANIC( "LOMAC/K: unexpected fget failure in wrap_readv.\n" );
    }
#endif
    handle_monitor_read( p_file->f_dentry );
    fput( p_file );

  } /* if readv succeeded */

  return( ret_val );

} /* wrap_readv() */


/* wrap_write()
 *
 * LOMAC attempts to provide the following guarantees:
 * o No process will write to a file, local-domain socket, or a FIFO with
 *   a level higher than its own.
 *   Exception: writes are always allowed on files like serial devices
 *   and pseudo-terminals.  See WRITE_EXEMPT macro for complete list of
 *   exempt files.
 * o If a process writes to a pipe, the pipe's level will be reduced to
 *   match the process's level. 
 *
 */

int wrap_write( unsigned int fd, char *buf, unsigned int count ) {

  struct file *p_file;       /* file struct corresponding to `fd' */
  int fd_is_pipe_flag;       /* set iff `fd' corresponds to an unnamed pipe */
  int ret_val = 0;           /* holds function return values */

#ifdef PARANOID
  global_wrapper_name_s = "write";
#endif

  if( !( p_file = fget( fd ) ) ) {
    return( -EBADF );   /* fail as original system call does */
  }

  /* Determine whether or not `fd' corresponds to an unnamed pipe. */
  fd_is_pipe_flag = IS_PIPE( p_file->f_dentry );

  /* If `fd' is not an unnamed pipe, we must perform mediation on the write */
  if( !fd_is_pipe_flag ) {
    if( ! handle_mediate_write( "write", p_file->f_dentry ) ) {
      ret_val =  -EACCES;  /* LOMAC indicates write should be denied. */
    }
  }

  fput( p_file );

  if( ret_val ) {
    return( ret_val );
  }

  /* LOMAC indicates write should be allowed.  Make system call. */
  ret_val = ((int (*)(unsigned int, char *, int))orig_write)( fd, buf, count );

  /* If the write succeeded, and if `fd' is an unnamed pipe, we must monitor */
  if( ( ret_val >= 0 ) && ( fd_is_pipe_flag ) ) {
    p_file = fget( fd );
#ifdef PARANOID
    if( !p_file ) {
      /* fget should not fail since write just succeeded. */
      PANIC( "LOMAC/K: unexpected fget failure in wrap_write.\n" );
    }
#endif
    monitor_pipe_write( current, p_file->f_dentry );
    fput( p_file );
  }

  return( ret_val );

} /* wrap_write() */


/* wrap_writev()
 *
 */

int wrap_writev( unsigned long fd, const struct iovec *vector, long count ) {

  struct file *p_file;       /* file struct corresponding to `fd' */
  int fd_is_pipe_flag;       /* set iff `fd' corresponds to an unnamed pipe */
  int ret_val = 0;           /* holds function return values */

#ifdef PARANOID
  global_wrapper_name_s = "writev";
#endif

  if( !( p_file = fget( fd ) ) ) {
    return( -EBADF );   /* fail as original system call does */
  }

  /* Determine whether or not `fd' corresponds to an unnamed pipe. */
  fd_is_pipe_flag = IS_PIPE( p_file->f_dentry );

  /* If `fd' is not an unnamed pipe, we must perform mediation on the write */
  if( !fd_is_pipe_flag ) {
    if( ! handle_mediate_write( "writev", p_file->f_dentry ) ) {
      ret_val = -EACCES;    /* LOMAC indicates write should be denied. */
    }
  }

  fput( p_file );

  if( ret_val ) {
    return( ret_val );
  }

  /* LOMAC indicates write should be allowed.  Make system call. */
  ret_val = ((int (*)(unsigned long, const struct iovec *, long))orig_writev)
                     ( fd, vector, count );

  /* If the write succeeded, and if `fd' is an unnamed pipe, we must monitor */
  if( ( ret_val >= 0 ) && ( fd_is_pipe_flag ) ) {
    p_file = fget( fd );
#ifdef PARANOID
    if( !p_file ) {
      /* fget should not fail since write just succeeded. */
      PANIC( "LOMAC/K: unexpected fget failure in wrap_writev.\n" );
    }
#endif
    monitor_pipe_write( current, p_file->f_dentry );
    fput( p_file );
  }

  return( ret_val );

} /* wrap_writev() */


/* wrap_pipe()
 *
 * LOMAC attempts to provide the following guarantees:
 * o A newly-created pipe will inherit the level of its creator.
 *
 */

int wrap_pipe( unsigned long *fildes ) {

  int ret_val;              /* holds return value */
  unsigned long fd_index;   /* file descriptor index for pipe inode */
  struct file *p_file;      /* points to file struct corresponding to pipe */

#ifdef PARANOID
  global_wrapper_name_s = "pipe";
#endif

  ret_val = ((int (*)(unsigned long *))orig_pipe)( fildes );

  /* If this was a successful pipe creation, set initial pipe "level" */
  if( !ret_val ) {

    /* The kernel's sys_pipe() call writes the appropriate file      *
     * descriptor values directly out to the calling process's file  *
     * table.  We must copy one of the file descriptors back into    *
     * kernel space to figure out what inode the kernel is using for *
     * the pipe.                                                     */
    copy_from_user( &fd_index, fildes, sizeof( long ) );
    p_file = fget( fd_index );
#ifdef PARANOID
    if( !p_file ) { 
      /* This situation shouldn't occur, because syscall just succeeded. */
      PANIC( "LOMAC/K: wrap_pipe got bad file descriptor.\n" );
    }
#endif

    monitor_pipe_create( current, p_file->f_dentry );
    fput( p_file );

  } /* if pipe was successfully created */

  return( ret_val );

} /* wrap_pipe() */


/* wrap_mknod()
 *
 * LOMAC attempts to provide the following guarantees:
 * o A low-level process cannot create a high-level FIFO.
 * o A low-level process can use mknod only to create FIFOs.
 * o A low-level process cannot create a FIFO in a high-level directory.
 *
 */

int wrap_mknod(const char *filename, int mode, dev_t dev) {

  struct dentry *p_dentry;     /* dentry for new FIFOs (ignored) */
  struct dentry *p_dir_dentry; /* dentry for new FIFO's parent directory */
  char *k_filename_s;          /* kernel-space copy of `filename' */
  char *k_canabspath_s;        /* kernel-space canabs copy of `filename' */
  int ret_val;                 /* holds function return values */

#ifdef PARANOID
  global_wrapper_name_s = "mknod";
#endif

  /* make `k_canabspath_s' a kernel-space copy of `filename' in canonical  *
   * absolute form.  Immediately call dput() on p_dentry and p_dir_dentry: *
   * we don't need the information they contain in this case.              */
  if( IS_ERR( ( k_filename_s = getname( filename ) ) ) ) {
    return( PTR_ERR( k_filename_s ) );
  }
  if( !( k_canabspath_s = (char *)__get_free_page( GFP_KERNEL ) ) ) {
    ret_val = -ENOMEM;
    goto out_putname;
  }
  ret_val = make_canabspath( k_filename_s, k_canabspath_s, 
			     &p_dir_dentry, &p_dentry );
  if( p_dentry ) {
    dput( p_dentry );  /* we ignore `p_dentry' - we don't need it */
  }
  if( ret_val  ) {
    goto out_free_page;
  }

  /* If `current' is trying to create a FIFO, make sure LOMAC will *
   * allow `current' to create a file with name `filename'.        */
  if( S_ISFIFO( mode ) ) {

    /* mediate */
    if( !( mediate_subject_path( "mknod", current, k_canabspath_s ) ) ) {
      ret_val = -EACCES;
      goto out_dput;
    }
    if( !( mediate_subject_object( "mknod", current, p_dir_dentry ) ) ) {
      ret_val = -EACCES;
      goto out_dput;
    }

  } else {

    /* If `current' wants to do something other than create a FIFO,   *
     * we must ensure that `current' is at the highest level.  David  *  
     * Wheeler of IDA has suggested a better scheme where low-level   *
     * processes can use mknod to create certain kinds of harmless    *
     * device special files, but this requires LOMAC to understand    *
     * device major and minor numbers.  Due to time constraints, this *
     * functionality will have to wait for a later version of LOMAC.  */

    if( !mediate_subject_at_lattr( "mknod", current, LATTR_HIGH ) ) {
      ret_val = -EACCES;
      goto out_dput;
    }

  } /* if/else current is creating a FIFO */

  TURN_ARG_CHECKS_OFF;
  ret_val = ((int (*)(const char *, int, dev_t))orig_mknod)
    ( k_canabspath_s, mode, dev );
  TURN_ARG_CHECKS_ON;

  /* If the mknod call succeeded in making a FIFO, prompt LOMAC to set *
   * the level of the inode it created to represent the new FIFO.      */
  if( S_ISFIFO( mode ) && ret_val >= 0 ) {

    /* Find the dir cache entry corresponding to the new FIFO *
     * and pass it to LOMAC's monitoring function.            */
    p_dentry = kernel_namei( k_canabspath_s );
    if( !IS_ERR( p_dentry ) ) {
      monitor_open( current, p_dentry );
      dput( p_dentry );
    } 
#ifdef PARANOID
    else {
      /* the kernel_namei() call shouldn't fail, since the mknod succeded. */
      PANIC( "LOMAC/K: unexpected kernel_namei() failure in wrap_mknod().\n" );
    }
#endif      

  } /* if mknod call succeeded in creating a new FIFO */

 out_dput:
  dput( p_dir_dentry );
 out_free_page:
  free_page( (unsigned long)k_canabspath_s );
 out_putname:
  putname( k_filename_s );
  return( ret_val );

} /* wrap_mknod() */


/* wrap_truncate()
 *
 * LOMAC attempts to provide the following guarantees:
 * o A process cannot tuncate a file, fifo, or local-domain socket with
 *   a level higher than its own.  
 *   Exception: truncates are always allowed on files that are exempt
 *              from write mediation, like serial devices and pseudo-
 *              terminals.  See WRITE_EXEMPT macro for complete list
 *              of exempt files.
 * Note that we don't check the dir's level here.  A low process can
 * truncate a low file in a high directory.
 */

int wrap_truncate( const char *path, unsigned long length ) {

  char *k_path_s;              /* kernel-space copy of `path' */
  char *k_canabspath_s;        /* kernel-space canabs copy of `path' */
  struct dentry *p_dentry;     /* dentry for `path' if one exists */
  struct dentry *p_dir_dentry; /* dentry for the parent directory of `path' */
  int ret_val;                 /* holds return values */

#ifdef PARANOID
  global_wrapper_name_s = "truncate";
#endif

  /* make `k_canabspath_s' a kernel-space copy of `filename' in canonical *
   * absolute form.  Make `p_dentry' point to the dentry for `filename'   *
   * if `filename' describes an existing file, and make p_dir_dentry      *
   * point to its parent directory.                                       */
  if( IS_ERR( ( k_path_s = getname( path ) ) ) ) {
    return( PTR_ERR( k_path_s ) );
  }
  if( !( k_canabspath_s = (char *)__get_free_page( GFP_KERNEL ) ) ) {
    ret_val = -ENOMEM;
    goto out_putname;
  }
  if( ( ret_val = make_canabspath( k_path_s, k_canabspath_s, 
				   &p_dir_dentry, &p_dentry ) ) ) {
    goto out_dputs;
  }

  /* The `current' process is trying to truncate the file named by *
   * `filename'.  This action is effectively a write, much like a  *
   * creat.  We will handle this write in the same way we handle   * 
   * writes due to creat.                                          */
  if( ! ( p_dentry && WRITE_EXEMPT( p_dentry ) ) ) { 

    /* mediate */
    if( !( mediate_subject_path( "truncate", current, k_canabspath_s ) ) ) {
      ret_val = -EACCES;
      goto out_dputs;
    }

  } /* if the file is not exempt from mediation */

  /* Make truncate system call. */ 
  TURN_ARG_CHECKS_OFF;
  ret_val = ((int (*)(const char *, unsigned long))orig_truncate)
    ( k_canabspath_s, length );
  TURN_ARG_CHECKS_ON;

 out_dputs:
  if( p_dir_dentry ) {
    dput( p_dir_dentry );
  }
  if( p_dentry ) {
    dput( p_dentry );
  }
  free_page( (unsigned long)k_canabspath_s );
 out_putname:
  putname( k_path_s );
  return( ret_val );

} /* wrap_truncate() */


/* wrap_ftruncate()
 *
 * LOMAC attempts to provide the following guarantees:
 * o A process cannot tuncate a file, fifo, or local-domain socket with
 *   a level higher than its own.
 *   Exception: truncates are always allowed on files that are exempt
 *              from write mediation, like serial devices and pseudo-
 *              terminals.  See WRITE_EXEMPT macro for complete list
 *              of exempt files.
 * o If a process truncates an unnamed pipe with a level higher than its
 *   own, the unnamed pipe's level will be reduced to match the process's
 *   level.
 */

int wrap_ftruncate( unsigned int fd, unsigned long length ) {

  struct file *p_file;       /* file struct corresponding to `fd' */
  int fd_is_pipe_flag;       /* set iff `fd' corresponds to an unnamed pipe */
  int ret_val = 0;           /* holds function return values */

#ifdef PARANOID
  global_wrapper_name_s = "ftruncate";
#endif

  if( !( p_file = fget( fd ) ) ) {
    return( -EBADF );   /* fail as original system call does */
  }

  /* Determine whether or not `fd' corresponds to an unnamed pipe. */
  fd_is_pipe_flag = IS_PIPE( p_file->f_dentry );

  /* If `fd' is not an unnamed pipe, we must perform mediation on the write */
  if( !fd_is_pipe_flag ) {
    if( ! handle_mediate_write( "ftruncate", p_file->f_dentry ) ) {
      ret_val = -EACCES;    /* LOMAC indicates write should be denied. */
    }
  }

  fput( p_file );

  if( ret_val ) {
    return( ret_val );
  }

  /* LOMAC indicates write should be allowed.  Make system call. */
  ret_val = ((int (*)(unsigned int, unsigned long))orig_ftruncate)
    ( fd, length );

  /* If the write succeeded, and if `fd' is an unnamed pipe, we must monitor */
  if( ( ret_val >= 0 ) && ( fd_is_pipe_flag ) ) {
    p_file = fget( fd );
#ifdef PARANOID
    if( !p_file ) {
      /* fget should not fail since write just succeeded. */
      PANIC( "LOMAC/K: unexpected fget failure in wrap_ftruncate.\n" );
    }
#endif
    monitor_pipe_write( current, p_file->f_dentry );
    fput( p_file );
  }

  return( ret_val );

} /* wrap_ftruncate() */


/* wrap_unlink()
 *
 * LOMAC attempts to provide the following guarantees:
 * o No process can unlink a file, fifo, or local-domain socket whose
 *   level is higher than its own.
 * o No process can unlink a file, fifo, or local-domain socket residing
 *   in a directory whose level is higher than the process's level.
 *
 */

int wrap_unlink( const char *pathname ) {

  char *k_pathname_s;          /* kernel-space copy of `pathname' */
  char *k_canabspath_s;        /* kernel-space canabs copy of `pathname' */
  struct dentry *p_dentry;     /* dentry for `pathname' if one exists */
  struct dentry *p_dir_dentry; /* dentry for `pathname's parent directory */
  int ret_val;                 /* holds return values from functions */

#ifdef PARANOID
  global_wrapper_name_s = "unlink";
#endif

  /* make `k_canabspath_s' a kernel-space copy of `pathname' in *
   * canonical absolute form.  Call dput() on `p_dentry' and    *
   * `p_dir_dentry' immediately; we don't need them.            */
  if( IS_ERR( k_pathname_s = getname( pathname ) ) ) {
    return( PTR_ERR( k_pathname_s ) );
  }
  if( !( k_canabspath_s = (char *)__get_free_page( GFP_KERNEL ) ) ) {
    ret_val = -ENOMEM;
    goto out_putname;
  }
  ret_val = make_canabspath( k_pathname_s, k_canabspath_s,
			     &p_dir_dentry, &p_dentry );
  if( p_dentry ) {
    dput( p_dentry );  /* we don't need this dentry */
  }
  if( ret_val ) {
    goto out_free_page;
  }

  /* The `current' process is trying to unlink the file named by  *
   * `filename'.  This action is effectively a write, much like a *
   * creat.  We will handle this write in the same way we handle  * 
   * writes due to creat, except we will not automatically allow  *
   * the unlinking of write-mediation-exempt files such as serial *
   * devices.                                                     */
  if( !( mediate_subject_path( "unlink", current, k_canabspath_s ) ) ) {
    ret_val = -EACCES;
    goto out_dput;
  }
  if( !( mediate_subject_object( "unlink", current, p_dir_dentry ) ) ) {
    ret_val = -EACCES;
    goto out_dput;
  }

  /* Make unlink system call. */ 
  TURN_ARG_CHECKS_OFF;
  /* ret_val = ((int (*)(const char *))orig_unlink)( k_canabspath_s ); */
  /* We're using `pathname' instead of `k_canabspath_s' as a temporary  *
   * strategy to avoid an annoying bug: make_canabspath always          *
   * translates symbolic links to their targets.  If you try to rm a    *
   * symbolic link to a directory, make_canabspath translates the link  *
   * to the directory, and when the unlink system call tries to unlink  *
   * the directory, it fails, complaining "is a directory".             *
   * make_canabspath needs to be fixed. ##                              */
  ret_val = ((int (*)(const char *))orig_unlink)( pathname );
  TURN_ARG_CHECKS_ON;

 out_dput:
  dput( p_dir_dentry );
 out_free_page:
  free_page( (unsigned long)k_canabspath_s );
 out_putname:
  putname( k_pathname_s );
  return( ret_val );

} /* wrap_unlink() */


/* wrap_rename()
 *
 * LOMAC attempts to provide the following guarantees:
 * o a process cannot rename a file if the file's level is higher
 *   than the process's level
 * o a process cannot rename a file if the new name for the file
 *   would cause it to have a higher level than the old name.
 *   (recall that file levels are determined by filename in this
 *    version of LOMAC).
 * o a process cannot rename a file residing in a directory with
 *   a level higher than the process's level.
 * o a process cannot rename a file to a new name that resides
 *   in a directory with a higher level than the process's.
 */

int wrap_rename( const char *oldname, const char *newname ) {

  struct dentry *p_dentry;     /* parm for make_canabspath(), ignored */
  struct dentry *p_old_dir_dentry;   /* parent dir for `oldname' */
  struct dentry *p_new_dir_dentry;   /* parent dir for `newname' */
  char *k_oldname_s;           /* kernel-space copy of `oldname' */
  char *k_newname_s;           /* kernel-space copy of `newname' */
  char *k_canabsoldname_s;     /* kernel copy of `oldname' in canabs form */
  char *k_canabsnewname_s;     /* kernel copy of `newname' in canabs form */
  int ret_val;                 /* holds values returned by functions */

#ifdef PARANOID
  global_wrapper_name_s = "rename";
#endif

  /* make `k_canabsoldname_s' a kernel-space copy of `oldname' in canonical *
   * absolute form.  Call dput() on p_dentry immediately; we don't need it. */
  if( IS_ERR( k_oldname_s = getname( oldname ) ) ) {
    return( PTR_ERR( k_oldname_s ) );
  }
  if( !( k_canabsoldname_s = (char *)__get_free_page( GFP_KERNEL ) ) ) {
    ret_val = -ENOMEM;
    goto out_putname_old;
  }
  ret_val = make_canabspath( k_oldname_s, k_canabsoldname_s,
			     &p_old_dir_dentry, &p_dentry );
  if( p_dentry ) {
    dput( p_dentry );
  }
  if( ret_val ) {
    goto out_free_page_old;
  }

  /* make `k_canabsnewname_s' a kernel-space copy of `newname' in canonical *
   * absolute form.  Call dput() on p_dentry immediately; we don't need it. */
  if( IS_ERR( k_newname_s = getname( newname ) ) ) {
    ret_val = PTR_ERR( k_newname_s );
    goto out_dput_old;
  }
  if( !( k_canabsnewname_s = (char *)__get_free_page( GFP_KERNEL ) ) ) {
    ret_val = -ENOMEM;
    goto out_putname_new;
  }
  ret_val = make_canabspath( newname, k_canabsnewname_s,
			     &p_new_dir_dentry, &p_dentry );
  if( p_dentry ) {
    dput( p_dentry );
  }
  if( ret_val ) {
    goto out_free_page_new;
  }

  /* make sure `current's level is at least as great as both directories */
  if( !( ( mediate_subject_object( "rename", current, p_old_dir_dentry ) ) &&
	 ( mediate_subject_object( "rename", current, p_new_dir_dentry ) )
	 ) ) {
    ret_val = -EACCES;
    goto out_dput_new;
  }

  /* First, ensure that `current's level is as least *
   * as great as the level of `k_canabsoldname_s'.     */
  /* Second, ensure that the level of the old path is *
   * at least as great as the level of the new path.  */
  if( !( ( mediate_subject_path( "rename", current, k_canabsoldname_s ) ) &&
	 ( mediate_path_path( "rename", current, k_canabsoldname_s,
			      k_canabsnewname_s ) ) ) ) {
    ret_val = -EACCES;
    goto out_dput_new;
  }

  /* mediation passed, found no errors, call original rename system call */
  TURN_ARG_CHECKS_OFF;
  ret_val = ((int (*)(const char *, const char *))orig_rename)
    ( k_canabsoldname_s, k_canabsnewname_s );
  TURN_ARG_CHECKS_ON;

 out_dput_new:
  dput( p_new_dir_dentry );
 out_free_page_new:
  free_page( (unsigned long)k_canabsnewname_s );
 out_putname_new:
  putname( k_newname_s );
 out_dput_old:
  dput( p_old_dir_dentry );
 out_free_page_old:
  free_page( (unsigned long)k_canabsoldname_s );
 out_putname_old:
  putname( k_oldname_s );
  return( ret_val );

} /* wrap_rename() */


/* wrap_link()
 *
 * LOMAC attempts to provide the following guarantees:
 * o a process cannot link to a file if the file's level is higher
 *   than the process's level
 * o a process can make a new link to an old file only if the file
 *   has the same level when named through both new and old links. 
 * o a process cannot create a link in a directory whose level is
 *   higher than the process's.
 */

int wrap_link( const char *oldname, const char *newname ) {

  struct dentry *p_dentry;      /* parm for make_canabspath(), ignored */
  struct dentry *p_old_dir_dentry;  /* `oldname' parent directory */
  struct dentry *p_new_dir_dentry;  /* `newname' parent directory */
  char *k_oldname_s;            /* kernel copy of `oldname' */
  char *k_newname_s;            /* kernel copy of `newname' */
  char *k_canabsoldname_s;      /* kernel copy of `oldname' in canabs form */
  char *k_canabsnewname_s;      /* kernel copy of `newname' in canabs form */
  int ret_val;                  /* holds values returned by functions */

#ifdef PARANOID
  global_wrapper_name_s = "link";
#endif

  /* make `k_canabsoldname_s' a kernel-space copy of `oldname' in canonical *
   * absolute form.  Call dput() on p_dentry immediately; we don't need it. */
  if( IS_ERR( ( k_oldname_s = getname( oldname ) ) ) ) {
    return( PTR_ERR( k_oldname_s ) );
  }
  if( !( k_canabsoldname_s = (char *)__get_free_page( GFP_KERNEL ) ) ) {
    ret_val = -ENOMEM;
    goto out_putname_old;
  }
  ret_val = make_canabspath( k_oldname_s, k_canabsoldname_s, 
			     &p_old_dir_dentry, &p_dentry );
  if( p_dentry ) {
    dput( p_dentry );
  }
  if( ret_val ) {
    goto out_free_page_old;
  }

  /* make `k_canabsnewname_s' a kernel-space copy of `newname' in canonical *
   * absolute form.  Call dput() on p_dentry immediately; we don't need it. */
  if( IS_ERR( ( k_newname_s = getname( newname ) ) ) ) {
    ret_val = PTR_ERR( k_newname_s );
    goto out_dput_old;
  }
  if( !( k_canabsnewname_s = (char *)__get_free_page( GFP_KERNEL ) ) ) {
    ret_val = -ENOMEM;
    goto out_putname_new;
  }
  ret_val = make_canabspath( k_newname_s, k_canabsnewname_s,
			     &p_new_dir_dentry, &p_dentry );
  if( p_dentry ) {
    dput( p_dentry );
  }
  if( ret_val ) {
    goto out_free_page_new;
  }

  /* Make sure `current's level is as least as high as the new dir's. */
  if( !mediate_subject_object( "link", current, p_new_dir_dentry ) ) {
    ret_val = -EACCES;
    goto out_dput_new;
  }

  /* First, ensure that `current's level is as least  *
   * as great as the level of `k_canabsoldname_s'.    */
  /* Second, ensure that the level of the old path is *
   * at least as great as the level of the new path.  */
  if( !( ( mediate_subject_path( "link", current, k_canabsoldname_s ) ) &&
	 ( mediate_path_path( "link", current, k_canabsoldname_s,
			      k_canabsnewname_s ) ) ) ) {
    ret_val = -EACCES;
    goto out_dput_new;
  }

  /* mediation passed, call original link system call */
  TURN_ARG_CHECKS_OFF;
  ret_val = ((int (*)(const char *, const char *))orig_link)
    ( k_canabsoldname_s, k_canabsnewname_s );
  TURN_ARG_CHECKS_ON;

 out_dput_new:
  dput( p_new_dir_dentry );
 out_free_page_new:
  free_page( (unsigned long)k_canabsnewname_s );
 out_putname_new:
  putname( k_newname_s );
 out_dput_old:
  dput( p_old_dir_dentry );
 out_free_page_old:
  free_page( (unsigned long)k_canabsoldname_s );
 out_putname_old:
  putname( k_oldname_s );
  return( ret_val );

} /* wrap_link */


/* wrap_socketcall()
 *
 */

int wrap_socketcall( int call, unsigned long *args ) {

#ifdef PARANOID
  global_wrapper_name_s = "socketcall";
#endif

  /* The kernel multiplexes many socket services in a single system     *
   * call: sys_socketcall.  Callers specify the service they want in    *
   * `call'.  LOMAC is concerned with only a few of these services.     *
   * The following switch sends calls to those services that LOMAC      *
   * cares about to the appropriate handling function.  The rest fall   *
   * through and are passed directly on to the kernel's sys_socketcall. */
  switch( call ) {
  case SYS_ACCEPT:
    return( handle_accept( call, args ) );
  case SYS_BIND:
    return( handle_bind( call, args ) );
  case SYS_CONNECT:
    return( handle_connect( call, args ) );
  case SYS_RECV:
  case SYS_RECVFROM:
  case SYS_RECVMSG:
    return( handle_recv( call, args ) );
  case SYS_SOCKETPAIR:
    return( handle_socketpair( call, args ) );
  }

  /* call original sys_socketcall() */
  return( ((int (*)(int, unsigned long *))orig_socketcall)( call, args ) );

} /* wrap_socketcall() */


/* wrap_kill()
 *
 * LOMAC attempts to provide the following guarantees:
 *
 * o Only processes at the highest possible level can send a signal
 *   to all of the processes on the system.
 * o A process cannot signal another process with a higher level.
 *
 */

int wrap_kill( int pid, int sig ) {

  int ret_val;   /* value returned by this function */
  struct task_struct *p_target = NULL;  /* task that is target of signal */

#ifdef PARANOID
  global_wrapper_name_s = "kill";
#endif

  /* We must perfrom different mediation depending on who the target
   * of the signal is.  The target is determined by the `pid' parm as
   * follows:
   * pid == 0   The `current' process is signalling its own process
   *            group.  We always allow this behavior - no mediation.
   * pid == -1  The `current' process is signalling all of the
   *            processes on the system.  We allow this only if
   *            the `current' process is at the highest possible level.
   * pid < -1
   * and
   * pid > 0    The `current' process is signalling another process
   *            or process group.  We allow this only if the `current'
   *            process's level is as least as great as the level of
   *            the target process or process group.  Note that LOMAC
   *            does the same mediation for processes and process
   *            groups because all processes in a given process group
   *            are always at the same level.
   */

  ret_val = -EACCES;   /* pessimistically assume we will deny operation */

  if( pid == 0 ) {

    /* The `current' process is signalling its own process group. *
     * We always allow this behavior - no mediation needed.       */
    ret_val = 0;

  } else if( pid == -1 ) {

    /* The process wants to signal all processes in the system. *
     * Allow this only if the process is at the highest level.  */
    if( mediate_subject_at_lattr( "kill", current, LATTR_HIGH ) ) {
      ret_val = 0;
    }

  } else {

    /* We handle the process and process group cases here.  If the    *
     * `current' process is trying to signal a process group ( pid <  *
     * -1 ), then we will set `p_target' to any process in that       *
     * process group.  On the other hand, if the `current' process is *
     * trying to signal another process, we will make `p_target'      *
     * point to the target process.  Once we have `p_target' properly *
     * set, we will call the appropriate LOMAC mediation function to  *
     * make a decision.                                               */

    if( pid < -1 ) {

      /* Signal pgrp -pid.  Make `p_target' point to any *
       * process in pgrp number -pid.                    */
      p_target = pgrp_to_task( -pid );

    } else {

      /* Signal process pid.  Make `p_target' point to process `pid'. */
      p_target = pid_to_task( pid );

    } /* if pgrp else process */

    /* If there was no such task, return same error as sys_kill(). */
    if( !p_target ) {
      return( -ESRCH );
    }

    /* Now that we have `p_target' set, perform mediation. */
    if( mediate_subject_subject( "kill", current, p_target ) ) {
      ret_val = 0;
    }

  } /* if/elseif/else various cases on `pid' */

  /* If mediation resulted in -EACCES, return immediately */
  if( ret_val ) {
    return( ret_val );
  }

  return( ((int (*)(int, int))orig_kill)( pid, sig ) );

} /* wrap_kill() */


/* wrap_mkdir()
 *
 * LOMAC attempts to provide the following guarantees:
 * o No process can make a directory with a level is higher than its own.
 * o A process cannot create a new directory inside an old directory
 *   whose level is higher than the process's level.
 */

int wrap_mkdir( const char *pathname, int mode ) {

  char *k_pathname_s;          /* kernel-space copy of `pathname' */
  char *k_canabspath_s;        /* kernel-space canabs copy of `pathname' */
  struct dentry *p_dentry;     /* dentry for `pathname' if one exists */
  struct dentry *p_dir_dentry; /* dentry for `pathname's parent directory */
  int ret_val;                 /* holds return values */

#ifdef PARANOID
  global_wrapper_name_s = "mkdir";
#endif

  /* make `k_canabspath_s' a kernel-space copy of `pathname' in canonical   *
   * absolute form.  Call dput() on p_dentry immediately; we don't need it. */
  if( IS_ERR( ( k_pathname_s = getname( pathname ) ) ) ) {
    return( PTR_ERR( k_pathname_s ) );
  }
  if( !( k_canabspath_s = (char *)__get_free_page( GFP_KERNEL ) ) ) {
    ret_val = -ENOMEM;
    goto out_putname;
  }
  ret_val = make_canabspath( k_pathname_s, k_canabspath_s,
			     &p_dir_dentry, &p_dentry );
  if( p_dentry ) {
    dput( p_dentry );
  }
  if( ret_val ) {
    goto out_free_page;
  }

  /* The `current' process is trying to create the directory named   *
   * by `pathname'.  This action is effectively a write, much like   *
   * a creat.  We will handle this write in the same way we handle   *
   * writes due to creat, except we will not allow write-exemptions, *
   * which are intended for serial devices, etc., not directories.   */
  if( !( mediate_subject_path( "mkdir", current, k_canabspath_s ) ) ) {
    ret_val = -EACCES;
    goto out_dput;
  }
  if( !( mediate_subject_object( "mkdir", current, p_dir_dentry ) ) ) {
    ret_val = -EACCES;
    goto out_dput;
  }

  /* Make mkdir system call. */ 
  TURN_ARG_CHECKS_OFF;
  ret_val = ((int (*)(const char *, int))orig_mkdir)( k_canabspath_s, mode );
  TURN_ARG_CHECKS_ON;

 out_dput:
  dput( p_dir_dentry );
 out_free_page:
  free_page( (unsigned long)k_canabspath_s );
 out_putname:
  putname( k_pathname_s );
  return( ret_val );

} /* wrap_mkdir() */


/* wrap_rmdir()
 *
 * LOMAC attempts to provide the following guarantees:
 * o No process can remove a directory with a level is higher than its own.
 * o No process can remove a directory from a parent directory with a 
 *   level higher than its own.
 */

int wrap_rmdir( const char *pathname ) {

  char *k_pathname_s;          /* kernel-space copy of `pathname' */
  char *k_canabspath_s;        /* kernel-space canabs copy of `pathname' */
  struct dentry *p_dentry;     /* dentry for `pathname' if one exists */
  struct dentry *p_dir_dentry; /* dentry for `pathname's parent directory */
  int ret_val;                 /* holds return values */

#ifdef PARANOID
  global_wrapper_name_s = "rmdir";
#endif

  /* make `k_canabspath_s' a kernel-space copy of `pathname' in canonical   *
   * absolute form.  Call dput() on p_dentry immediately; we don't need it. */
  if( ( IS_ERR( k_pathname_s = getname( pathname ) ) ) ) {
    return( PTR_ERR( k_pathname_s ) );
  }
  if( !( k_canabspath_s = (char *)__get_free_page( GFP_KERNEL ) ) ) {
    ret_val = -ENOMEM;
    goto out_putname;
  }
  ret_val = make_canabspath( k_pathname_s, k_canabspath_s,
			     &p_dir_dentry, &p_dentry );
  if( p_dentry ) {
    dput( p_dentry );
  }
  if( ret_val ) {
    goto out_free_page;
  }

  /* The `current' process is trying to remove the directory named   *
   * by `pathname'.  This action is effectively a write, much like   *
   * a creat.  We will handle this write in the same way we handle   *
   * writes due to creat, except we will not allow write-exemptions, *
   * which are intended for serial devices, etc., not directories.   */
  if( !( mediate_subject_path( "rmdir", current, k_canabspath_s ) ) ) {
    ret_val = -EACCES;
    goto out_dput;
  }
  if( !( mediate_subject_object( "rmdir", current, p_dir_dentry ) ) ) {
    ret_val = -EACCES;
    goto out_dput;
  }

  /* Make rmdir system call. */ 
  TURN_ARG_CHECKS_OFF;
  ret_val = ((int (*)(const char *))orig_rmdir)( k_canabspath_s );
  TURN_ARG_CHECKS_ON;

 out_dput:
  dput( p_dir_dentry );
 out_free_page:
  free_page( (unsigned long)k_canabspath_s );
 out_putname:
  putname( k_pathname_s );
  return( ret_val );

} /* wrap_rmdir() */


/* wrap_mount()
 *
 * LOMAC attempts to provide the following guarantees:
 * o Only highest-level processes can mount.
 *
 */

int wrap_mount( char *dev_name, char *dir_name, char *type,
		unsigned long new_flags, void *data ) {

#ifdef PARANOID
  global_wrapper_name_s = "mount";
#endif

  if( !mediate_subject_at_lattr( "mount", current, LATTR_HIGH ) ) {
    return( -EACCES );
  }

  return( ((int (*)( char *, char *, char *, unsigned long, void *))orig_mount)
	  ( dev_name, dir_name, type, new_flags, data ) );

} /* wrap_mount() */


/* wrap_umount()
 *
 * LOMAC attempts to provide the following guarantees:
 * o Only highest-level processes can umount.
 *
 */

int wrap_umount( char *name, int flags ) {

#ifdef PARANOID
  global_wrapper_name_s = "umount";
#endif

  if( !mediate_subject_at_lattr( "umount", current, LATTR_HIGH ) ) {
    return( -EACCES );
  }

  return( ((int (*)( char *, int ))orig_umount)( name, flags ) );

} /* wrap_mount() */


/* wrap_reboot()
 *
 * LOMAC attempts to provide the following guarantees:
 * o Only highest-level processes can reboot.
 *
 */

int wrap_reboot( int magic1, int magic2, int cmd, void *arg ) {

#ifdef PARANOID
  global_wrapper_name_s = "reboot";
#endif

  if( !mediate_subject_at_lattr( "reboot", current, LATTR_HIGH ) ) {
    return( -EACCES );
  }

  return( ((int (*)( int, int, int, void * ))orig_umount)
	  ( magic1, magic2, cmd, arg ) );

} /* wrap_reboot() */


int wrap_create_module( const char *name_user, size_t size ) {

#ifdef PARANOID
  global_wrapper_name_s = "create_module";
#endif

  if( !mediate_subject_at_lattr( "create_module", current, LATTR_HIGH ) ) {
    return( -EACCES );
  }

  return( ((int (*)( const char *, size_t ))orig_create_module)( name_user, 
								 size ) );

} /* wrap_create_module() */


int wrap_init_module( const char *name_user, struct module *mod_user ) {

#ifdef PARANOID
  global_wrapper_name_s = "init_module";
#endif

  if( !mediate_subject_at_lattr( "init_module", current, LATTR_HIGH ) ) {
    return( -EACCES );
  }

  return( ((int (*)( const char *, struct module * ))orig_init_module)
	  ( name_user, mod_user ) );

} /* wrap_init_module() */


int wrap_delete_module( const char *name_user ) {

#ifdef PARANOID
  global_wrapper_name_s = "delete_module";
#endif

  if( !mediate_subject_at_lattr( "delete_module", current, LATTR_HIGH ) ) {
    return( -EACCES );
  }

  return( ((int (*)( const char * ))orig_delete_module)
	  ( name_user ) );

} /* wrap_delete_module() */



/**************************************************************************
 *
 * some utility functions...
 *
 **************************************************************************/


/* handle_monitor_read()
 *
 * in:     p_dentry   - dentry for file being read by read() or readv()
 * out:    nothing
 * return: nothing
 *
 *  This is a convenience function for wrap_read() and wrap_readv() -
 * it encapsulates some common code that they both need.
 *
 */

void handle_monitor_read( struct dentry *p_dentry ) {

#ifdef PARANOID  
  /* This function is called after the read or readv system call has   *
   * completed.  If the call failed, this function should not be       *
   * called.  Consequently, this function assumes that the read/v call *
   * succeeded, and `p_dentry' is a valid dentry.  These assertions    *
   * test this assumption.                                             */
  if( !p_dentry ) {
    PANIC( "LOMAC/K: handle_monitor_read() called on null dentry\n" );
  }
  if( !(p_dentry->d_inode) ) {
    PANIC( "LOMAC/K: handle_monitor_read() called on dentry with no inode\n" );
  }
#endif

  /* Processes use the read system calls to access a variety of        *
   * abstractions that the kernel represents with inodes.  LOMAC       *
   * considers some of these inode abstractions (like files) objects,  *
   * and others (like unnamed pipes) non-objects.  Even among those    *
   * inode abstractions that are objects, LOMAC uses different methods *
   * of determining their levels.  For example, the level of a file    *
   * object depends on its name in the filesystem, but the level of a  *
   * non-local domain socket depends on the hardware device it least   *
   * recently read from.  The following if statement                   *
   * distinguishes between all the kinds of inode abstractions and     *
   * objects so LOMAC can moderate each kind appropriately.            */

  if( IS_NETWORK_SOCKET( p_dentry ) ) {

    /* non-local domain socket */
    monitor_read_nic( current, NULL );

  } else {
    
    /* file, FIFO, unnamed pipe, UNIX (local) domain socket */
    monitor_read_object( current, p_dentry );

  }

} /* handle_monitor_read() */


/* handle_mediate_write()
 *
 * in:     op_s     - string describing operation, like "write" or "truncate"
 *         p_dentry - dentry for file being written by write() or writev()
 * out:    nothing
 * return: value    condition
 *         -----    ---------
 *           0      caller should prevent write
 *           1      caller should permit write
 *
 *  This is a convenience function for wrap_write() and wrap_writev() -
 * it encapsulates some common code that they both need.
 *
 */

int handle_mediate_write( const char *op_s, struct dentry *p_dentry ) {

#ifdef PARANOID  
  if( !op_s ) {
    PANIC( "LOMAC/K: handle_mediate_write() called with null op_s\n" );
  }
  /* We're relying on the caller to ensure that `p_dentry' is valid. */
  if( !p_dentry ) {
    PANIC( "LOMAC/K: handle_mediate_write() called on null dentry\n" );
  }
  if( !(p_dentry->d_inode) ) {
    PANIC( "LOMAC/K: handle_mediate_write() "
	   "called on dentry with no inode\n" );
  }

  /* LOMAC never prevents a write to an unnamed pipe.  This function *
   * should not be called on unnamed pipes.                          */
  if( IS_PIPE( p_dentry ) ) {
    PANIC( "LOMAC/K: handle_mediate_write() called on pipe\n" );
  }
#endif


  /* Processes use the write system calls to access a variety of       *
   * abstractions that the kernel represents with inodes.  LOMAC       *
   * considers some of these inode abstractions (like files) objects,  *
   * and others (like unnamed pipes) non-objects.  Even among those    *
   * inode abstractions that are objects, LOMAC uses different methods *
   * of determining their levels.  For example, the level of a file    *
   * object depends on its name in the filesystem, but the level of a  *
   * non-local domain socket depends on the hardware device it least   *
   * recently read from.  The following nested if statement            *
   * distinguishes between all the kinds of inode abstractions and     *
   * objects so LOMAC can moderate each kind appropriately.            */

  if( IS_NETWORK_SOCKET( p_dentry ) ) {
    /* non-local domain socket.  LOMAC always allows these writes. */
    return( 1 );  /* allow write */

  } else {

    /* FIFO or file */
    if( WRITE_EXEMPT( p_dentry ) ) {
      
      /* Exempt files - we always allow writes to serial devices, *
       * ttys, pseudoterminals, /dev/null.                        */
      return( 1 ); /* allow write */

    } else {

      /* non-exempt files */

      return( mediate_subject_object( op_s, current, p_dentry ) );

    } /* if/else exempt file */

  } /* else file */

  /* Eventually, we will handle writes on other kinds of objects here. */
  return( 1 );  /* allow... for now! ## */

} /* handle_mediate_write() */


/* handle_accept()
 *
 *
 */

int handle_accept( int call, unsigned long *args ) {

  unsigned long arg_array[ 3 ];  /* kernel-space copy of `args' */
  int server_socket_fd;          /* server socket file descriptor */
  int session_socket_fd;         /* session socket file descriptor */
  struct file *p_server_socket;  /* `current' wants to accept on this */
  struct file *p_session_socket; /* session socket created if accept works */

  /* The arguments for sys_accept are as follows:
   * args[ 0 ] (in) fd of server socket on which `current' is doing the accept
   * args[ 1 ] (out) peer sockaddr
   * args[ 2 ] (out) length of peer sockaddr
   *
   * Copy all three arguments into the kernel-space array `arg_array'. */
  if( copy_from_user( arg_array, args, ( sizeof( unsigned long ) * 3 ) ) ) {
    return( -EFAULT );
  }
  server_socket_fd = (int)(arg_array[ 0 ]);

  /* Since we'll be turning arg checking off during the actual system *
   * call, we must verify that args[ 1 ] and args[ 2 ] are addresses  *
   * in user-space now.                                               */
  if( !( ( access_ok( VERIFY_WRITE, args[ 1 ], sizeof( unsigned long ) ) ) &&
	 ( access_ok( VERIFY_WRITE, args[ 2 ], sizeof( unsigned long ) ) )
	 ) ) {
    return( -EFAULT );
  }

  /* Call original sys_socketcall().  It should return fd of session socket */
  TURN_ARG_CHECKS_OFF;
  session_socket_fd = 
    ((int (*)(int, unsigned long *))orig_socketcall)( call, arg_array );
  TURN_ARG_CHECKS_ON;

  /* Some of the parms in `arg_array' are "out" parms - they pass     *
   * information back to the caller.  So, we have to copy `arg_array' *
   * back out to the user-space `args' array.                         */
  if( copy_to_user( args, arg_array, ( sizeof( unsigned long ) * 3 ) ) ) {
    return( -EFAULT );
  }

  /* We're concerned only with successful accepts *
   * on UNIX (local) domain sockets.              */
  if( session_socket_fd >= 0 ) {

    if( !( p_server_socket = fget( server_socket_fd ) ) ) {
      /* we expect no error since sys_socketcall() succeeded. */
      PANIC( "LOMAC: unable to get server socket in handle_accept.\n" );
    }
    
    if( IS_LOCAL_SOCKET( p_server_socket->f_dentry ) ) {

      /* If this was a successful accept() call to a UNIX-domain socket, *
       * then the kernel has created a new session socket for handling   *
       * the accepted connection.  We must make `p_session_socket' point *
       * to this new session socket, and set its level to match          *
       * the server socket's level.                                      */

      if( !( p_session_socket = fget( session_socket_fd ) ) ) {
	/* we expect no error since sys_socketcall() succeeded. */
	PANIC( "LOMAC: unable to get session socket in handle_accept.\n" );
      }
      
      monitor_unix_socket_accept_connect( current, p_server_socket->f_dentry,
					  p_session_socket->f_dentry );

      fput( p_session_socket );

    } /* if server socket is a local socket */

    fput( p_server_socket );

  } /* if socketcall succeeded */

  return( session_socket_fd );

} /* handle_accept() */


/* handle_bind()
 *
 * LOMAC attempts to provide the following guarantees:
 * o no process can bind a UNIX domain socket to a name in the filesystem
 *   whose level is higher than the process's level.
 * o no process can use bind to create a name in a directory whose
 *   level is higher than the process's level.
 *
 */

int handle_bind( int call, unsigned long *args ) {

  unsigned long arg_array[ 3 ];   /* kernel-space copy of args[ 0..2 ] */
  char k_sockaddr[MAX_SOCK_ADDR]; /* kernel-space buffer for socket addrs */
  struct sockaddr *p_sockaddr;    /* generic sockaddr pointer */
  int socket_is_local_flag = 0;   /* set iff socket is a local-domain socket */
  int name_is_abstract_flag = 0;  /* set iff socket has abstract name */
  char *k_canabspath_s = NULL;    /* kernel-space copy of name for binding */
  struct file *p_socket;          /* `current' wants to bind this socket */
  struct dentry *p_dentry;        /* dentry of bound name in filesystem */
  struct dentry *p_dir_dentry = NULL; /* dentry of bound name's parent dir */
  int ret_val = 0;                /* value returned by function calls */

  /* The arguments to SYS_BIND are as follows:
   *  args[ 0 ]  (in) fd of socket
   *  args[ 1 ]  (in) address of name to bind to
   *  args[ 2 ]  (in) length of name
   *
   * Copy all three arguments into the kernel-space array `arg_array'. */
  if( copy_from_user( arg_array, args, ( sizeof( unsigned long ) * 3 ) ) ) {
    return( -EFAULT );
  }

  /* Copy the sockaddr pointed to by `arg_array[ 1 ]' into `k_sockaddr'.  *
   * Use the length in `arg_array[ 2 ]' to tell how many bytes to copy.   */
  if( ( ret_val = copy_sockaddr_to_kernel( (void *)(arg_array[ 1 ]),
					   arg_array[ 2 ], 
					   k_sockaddr ) ) ) {
    return( ret_val );
  }

  /* The kernel-space `arg_array[ 1 ]' presently points to the       *
   * user-space sockaddr buffer.  Make it point to the kernel-space  *
   * copy in `k_sockaddr'.  Make `p_sockaddr_un' point to this, too. */
  arg_array[ 1 ] = (unsigned long)k_sockaddr;
  p_sockaddr = (struct sockaddr *)k_sockaddr;

  /* Examine `*p_sockaddr' and set `socket_is_local_flag' *
   * and `name_is_abstract' flags appropriately.          */
  if( p_sockaddr->sa_family == AF_UNIX ) {
    socket_is_local_flag = 1;
    if( ((struct sockaddr_un *)p_sockaddr)->sun_path[ 0 ] == '\0' ) {
      name_is_abstract_flag = 1;
    }
  }

  /* If this is a local (unix) domain socket, we must mediate.  For *
   * now, we'll mediate only if the name is in the filesystem's     *
   * namespace, not the absrtact namespace that was introduced in   *
   * the 2.2 kernels.                                               */

  if( socket_is_local_flag && !name_is_abstract_flag ) {

    /* make `k_canabspath_s' an absolute canonical-form copy *
     * of the pathname in `k_addr_un.sun_path'.              */
    if( !( k_canabspath_s = (char *)__get_free_page( GFP_KERNEL ) ) ) {
      return( -ENOMEM );
    }
    ret_val = make_canabspath( ((struct sockaddr_un *)p_sockaddr)->sun_path,
			       k_canabspath_s, &p_dir_dentry, &p_dentry );
    if( p_dentry ) {
      dput( p_dentry );  /* don't need this */
    }
    if( ret_val ) {
      goto out;
    }

    /* mediate */
    if( !( mediate_subject_path( "bind", current, k_canabspath_s ) ) ) {
      ret_val = -EACCES;
      goto out;
    }
    if( !( mediate_subject_object( "bind", current, p_dir_dentry ) ) ) {
      ret_val = -EACCES;
      goto out;
    }

    /* copy our canonical-absolute name into `*p_sockaddr' so the *
     * original bind system call will use our canonicalized path. */
    /* UNIX_PATH_MAX = 108 in un.h is much shorter than PATH_MAX. */
    strncpy( ((struct sockaddr_un *)p_sockaddr)->sun_path,
	     k_canabspath_s, UNIX_PATH_MAX );

  } /* if we need to mediate */

  /* Call original sys_socketcall using our kernel-space copy of the args. */
  TURN_ARG_CHECKS_OFF;
  ret_val = ((int (*)(int, unsigned long *))orig_socketcall)( call, 
							      arg_array );
  TURN_ARG_CHECKS_ON;

  /* We're concerned only with successful binds on local domain sockets. */
  if( ( !ret_val ) && socket_is_local_flag ) {

    p_socket = fget( (int)(arg_array[ 0 ]) );
#ifdef PARANOID
    if( !p_socket ) {
      /* fget should not fail since sys_socketcall succeeded */
      PANIC( "LOMAC/K: fget failed in handle_bind.\n" );
    }
#endif

    /* We need to monitor this bind differently depending on whether     *
     * the bound name is in the filesystem or in the abstract namespace. */
    if( name_is_abstract_flag ) {

      /* The binding has created a name in the abstract namespace. We *
       * must set `p_socket's level appropriately.             */
      monitor_unix_socket_abstract( current, p_socket->f_dentry );

    } else {

      /* The binding has created a name in some filesystem namespace, *
       * represented by its own inode.  Make `p_dentry' point to      *
       * this file's dentry and set its level appropriately.          */
      p_dentry = socket_name( p_socket->f_dentry );
      if( !p_dentry ) {
	/* we expect no failures since sys_socketcall() succeeded. */
	PANIC( "LOMAC: can't find socket peer in handle_bind.\n" );
      }

      monitor_unix_socket_bind( current, p_socket->f_dentry, p_dentry );

    } /* if/else abstract/filesystem namespace */

    fput( p_socket );

  } /* if we need to monitor bind */

 out:
  if( p_dir_dentry ) {
    dput( p_dir_dentry );
  }
  if( k_canabspath_s ) {
      free_page( (unsigned long)k_canabspath_s );
  }

  return( ret_val );

} /* handle_bind() */


/* handle_connect()
 *
 *
 */

int handle_connect( int call, unsigned long *args ) {

  unsigned long arg_array[ 3 ];    /* kernel-space copy of `args' */
  int client_socket_fd;            /* client socket file descriptor */
  struct file *p_client_socket;    /* `current' wants to connect on this */
  struct dentry *p_session_socket; /* session socket for connection */
  int ret_val;                     /* value returned by function calls */

  /* The arguments to SYS_BIND are as follows:
   *  args[ 0 ]  (in) fd of socket current is doing connect on
   *  args[ 1 ]  (in) address to connect to
   *  args[ 2 ]  (in) length of address
   *
   * Copy all three arguments into the kernel-space array `arg_array'. */
  if( copy_from_user( arg_array, args, ( sizeof( unsigned long ) * 3 ) ) ) {
    return( -EFAULT );
  }
  client_socket_fd = (int)arg_array[ 0 ];

  /* Call original sys_socketcall().  It should return fd of session socket */
  TURN_ARG_CHECKS_OFF;
  ret_val = ((int (*)(int, unsigned long *))orig_socketcall)
    ( call, arg_array );
  TURN_ARG_CHECKS_ON;

  /* We're concerned only with successful connects on local domain sockets. */
  if( !ret_val ) {

    p_client_socket = fget( client_socket_fd );
#ifdef PARANOID
    if( !p_client_socket ) {
      /* fget should not fail since socketcall succeeded */
      PANIC( "LOMAC/K:  fget failed in handle_connect.\n" );
    }
#endif

    if( IS_LOCAL_SOCKET( p_client_socket->f_dentry ) ) {

      /* sys_connect() links the client's socket to the socket      *
       * created by sys_accept() in the server.  The implementation *
       * of the link has changed a bit from the 2.0 Linux kernels.  *
       * We must set the level of the client's socket to match the  *
       * level of the server's session socket that was created by   *
       * sys_accept().                                              */
      p_session_socket = socket_peer( p_client_socket->f_dentry );
      if( !p_session_socket ) {

	/* this indicates a connect to a socket bound to an abstract  *
	 * name, rather than a name in the filesystem namespace.  We  *
	 * (mis)handle these connects differently from normal         *
	 * connects.  It would probably be better to detect this case *
	 * by examining the address parameter.                        */
	monitor_unix_socket_abstract( current, p_client_socket->f_dentry );

      } else {

	monitor_unix_socket_accept_connect( current, p_session_socket,
					    p_client_socket->f_dentry );
      }

    } /* if local socket */

    fput( p_client_socket );

  } /* if socketcall succeeded */

  return( ret_val );

} /* handle_connect() */


/* handle_recv()
 *
 * This function has a TOC/TOU error in its handling of `args'.
 *
 */

int handle_recv( int call, unsigned long *args ) {

  unsigned long arg;            /* kernel-space copy of args[ 0 ] */
  int socket_fd;                /* socket file descriptor */
  struct file *p_socket;        /* `current' wants to recv on this socket */
  int ret_val;                  /* value returned by function calls */

  /* The client socket's fd number is in args[ 0 ].  Use it to get *
   * dentry for socket.                                            */
  copy_from_user( &arg, args, sizeof( unsigned long ) );
  socket_fd = (int)arg;

  /* We'll proceed without checking for errors - if there were errors, *
   * the following call to the real sys_socketcall will catch them.    */
  
  /* Call original sys_socketcall().  It should return fd of session socket */
  ret_val = ((int (*)(int, unsigned long *))orig_socketcall)( call, args );

  /* We're concerned only with successful recv's on local domain sockets. */
  if( ret_val >= 0 ) {

    p_socket = fget( socket_fd );
#ifdef PARANOID
    if( !p_socket ) {
      /* fget should not fail since socketcall just succeeded */
      PANIC( "LOMAC/K: fget failed in handle_recv.\n" );
    }
#endif

    if( IS_LOCAL_SOCKET( p_socket->f_dentry ) ) {
      handle_monitor_read( p_socket->f_dentry );
    }

    fput( p_socket );

  } /* if socketcall succeeded */

  return( ret_val );

} /* handle_recv() */


/* handle_socketpair()
 *
 * This code sets all UNIX-domain sockets created with socketpair to
 * LOMAC_LOWEST_LEVEL.  This is a poor solution, since it prevents
 * high-level processes from having high-level socketpairs.
 *
 * This function has a TOC/TOU error in its handling of `args'.
 *
 */

int handle_socketpair( int call, unsigned long *args ) {

  unsigned long arg_array[ 4 ];         /* kernel-space copy of args */
  int *sock_fds;                        /* fds for sockets in pair */
  struct file *sockets[ 2 ];            /* sockets in pair */
  int ret_val;                          /* value returned by function calls */

  /* Call original sys_socketcall(). */
  ret_val = ((int (*)(int, unsigned long *))orig_socketcall)( call, args );

  /* Return immediately if the kernel failed to create a socket pair */
  if( ret_val ) {
    return( ret_val );
  }

  /* The arguments for SYS_SOCKETCALL are as follows:
   * args[ 0 ] family
   * args[ 1 ] type
   * args[ 2 ] protocol
   * args[ 3 ] int * to the first of two ints.  The kernel's original
   * SYS_SOCKETPAIR code writes the fds for the new pair of sockets
   * to this location.
   */

  copy_from_user( arg_array, args, ( sizeof( unsigned long ) * 4 ) );
  
  /* we need to monitor only UNIX (local) domain sockets.  Return *
   * if this is not a UNIX (local) domain socketpair.             */
  if( (int)(arg_array[ 0 ]) != AF_UNIX ) {
    return( ret_val );
  }

  sock_fds = (int *)(arg_array[ 3 ]);
  sockets[ 0 ] = fget( sock_fds[ 0 ] );
  sockets[ 1 ] = fget( sock_fds[ 1 ] );
#ifdef PARANOID
  if( !( sockets[ 0 ] && sockets[ 1 ] ) ) {
    PANIC( "LOMAC/K: null socket file pointers in handle_socketpair.\n" );
  }
#endif

  monitor_unix_socketpair( current, sockets[ 0 ]->f_dentry, 
			   sockets[ 1 ]->f_dentry );  

  fput( sockets[ 1 ] );
  fput( sockets[ 0 ] );
  return( ret_val );

} /* handle_socketpair() */


