/*************************************************************************
 *
 * lomac_mediate.c
 *
 * LOMAC - Low Water-Mark Mandatory Access Control for Linux
 * Copyright (c) 1999, 2000, 2001, 2002 Networks Associates
 * Technology, Inc.  All rights reserved.
 * 
 * This file is part of LOMAC.
 *
 * LOMAC is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * LOMAC is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with LOMAC; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *
 * This file contains functions that make access control decisions
 * concerning wether or not given system calls should be allowed
 * or denied.  This activity is called "mediation".  These functions
 * generally consider both the parameters passed to a system call
 * and the current internal state of LOMAC in the course of making
 * a decision.  However, they do not modify these parameters or
 * LOMAC's internal state.  Functions for modifying LOMAC's internal
 * state can be found in lomac_monitor.c.
 *
 *
 *************************************************************************/

#include "kernel_interface.h"
#include "lomac_mediate.h"
#include "lomac_plm.h"
#include "lomac_log.h"

/* mediate_subject_subject()
 *
 * in:     op_s          - name of operation to mediate
 *         p_subject_one - subject one
 *         p_subject_two - subject two
 * out:    nothing
 * return: value   condition
 *         -----   ---------
 *           0     caller should deny operation
 *           1     caller should allow operation
 *
 *     This function returns 1 if `p_subject_one's level is at least
 * as great as `p_subject_two's level.  Otherwise, it logs a permission
 * failure on operation `op_s' and returns 0.
 *
 */

int mediate_subject_subject( const char *op_s,
			     const lomac_subject_t *p_subject_one,
			     const lomac_subject_t *p_subject_two ) {

  lattr_t lattr_one;     /* lattr of `p_subject_one' */
  lattr_t lattr_two;     /* lattr of `p_subject_two' */
  int ret_val;           /* result to return to caller */

#ifdef PARANOID
  if( !op_s ) {
    PANIC( "LOMAC: null op_s string passed to mediate_subject_subject()\n" );
  }
  if( !( p_subject_one && p_subject_two ) ) {
    PANIC( "LOMAC: null subject pointer passed "
	   "to mediate_subject_subject()\n" );
  }
#endif

#ifdef NO_MEDIATION
  ret_val = 1;        /* no denials, just logging */
#else
  ret_val = 0;        /* pessimistically assume deny */
#endif

  get_subject_lattr( p_subject_one, &lattr_one );
  get_subject_lattr( p_subject_two, &lattr_two );

  if( LEVEL_LEQ( lattr_two.level, lattr_one.level ) ) {

    ret_val = 1;      /* OK, allow */

  } else {

    /* log permission failure */
    if( verbose & VERBOSE_DEMOTE_DENY ) {
      log_start();
      log_append_string( "LOMAC: denied level-" );
      log_append_int( lattr_one.level );
      log_append_string( " proc " );
      log_append_subject_id( p_subject_one );
      log_append_string( op_s );
      log_append_string( " to level-" );
      log_append_int( lattr_two.level );
      log_append_string( " proc " );
      log_append_subject_id( p_subject_two );
      log_append_string( "'s subject\n" );
      log_print();
    }

  } /* if/else allow/deny */

#ifdef PARANOID
  if( !( ( ret_val == 0 ) || ( ret_val == 1 ) ) ) {
    PANIC( "LOMAC: bad return value in mediate_subject_subject()\n" );
  }
#endif

  return( ret_val );

} /* mediate_subject_subject() */


/* mediate_subject_path()
 * 
 * in:     op_s      - string describing operation, like "open" or "mknod".
 *         p_subject - subject performing operation on name `path_s'
 *         path_s    - name of object.  We can't pass a lomac_object_t *
 *                     here because the named object may not exist, and
 *                     possibly should never exist.
 * out:    nothing
 * return: value    condition
 *         -----    ---------
 *           0      Caller should prevent operation
 *           1      Caller should permit operation
 *
 * This function returns 1 if the level of `p_subject' is at least
 * as great as the level the PLM associates with the object name
 * `path_s'.  Otherwise, it returns 0 and logs a permission failure
 * on operation `op_s'.
 *
 * This function is used to mediate open, creat, mknod, truncate, 
 * link, unlink, mkdir.
 *
 */

int mediate_subject_path( const char *op_s, 
			  const lomac_subject_t *p_subject, 
			  const char *path_s ) {

  lattr_t subject_lattr;            /* holds lattr of `p_subject' */
  lattr_t path_lattr;               /* holds lattr PLM gives `path_s' */
  int ret_val;                      /* value to return to caller */

#ifdef PARANOID
  if( !op_s ) {
    PANIC( "LOMAC: null op_s pointer passed to mediate_subject_path()\n" );
  }
  if( !p_subject ) {
    PANIC( "LOMAC: null subject pointer passed to mediate_subject_path()\n" );
  }
  if( !path_s ) {
    PANIC( "LOMAC: null path_s pointer passed to mediate_subject_path()\n" );
  }
#endif

#ifdef NO_MEDIATION
  ret_val = 1;           /* allow operation regardless of decision */
#else
  ret_val = 0;           /* pessimistically assume deny */
#endif

  /* Get the levels of `p_subject' and `path_s'.  Deny operation *
   * if `path_s's level is higher that `p_subject's.             */ 
  get_subject_lattr( p_subject, &subject_lattr );
  get_pathname_lattr( path_s, &path_lattr );

  if( LEVEL_LEQ( path_lattr.level, subject_lattr.level ) ) {
    
    ret_val = 1;         /* allow operation */

  } else {

    /* log permission denial */
    if( verbose & VERBOSE_DEMOTE_DENY ) {
      log_start();
      log_append_string( "LOMAC: " );
      log_append_subject_id( p_subject );
      log_append_string( " level " );
      log_append_int( subject_lattr.level );
      log_append_string( " denied " );
      log_append_string( op_s );
      log_append_string( " of " );
      log_append_string( path_s );
      log_append_string( " level " );
      log_append_int( path_lattr.level );
      log_append_string( "\n" );
      log_print();
    }

  } /* if/else allow/deny */

#ifdef PARANOID
  if( !( ( ret_val == 0 ) || ( ret_val == 1 ) ) ) {
    PANIC( "LOMAC: bad return value in mediate_subject_path()\n" );
  }
#endif

  return( ret_val );

} /* mediate_subject_path() */


/* mediate_subject_object()
 *
 * in:     op_s      - string describing operation, like "write" or "writev"
 *         p_subject - subject trying to operate on `p_object'.
 *         p_object  - object that `p_subject' is trying to operate on.
 * out:    nothing
 * return: value    condition
 *         -----    ---------
 *           0      Caller should prevent operation
 *           1      Caller should permit operation
 * 
 * This function returns 1 if the level of `p_object' is less than or
 * equal to the level of `p_subject'.  Otherwise, it returns 0 and
 * logs a permission denial on `op_s'.
 *
 * This function allows LOMAC to mediate write and writev system calls.
 *
 */

int mediate_subject_object( const char *op_s, 
			    const lomac_subject_t *p_subject, 
			    const lomac_object_t *p_object ) {

  lattr_t subject_lattr;     /* lattr of `p_subject' */
  lattr_t object_lattr;      /* lattr of `p_object' */
  int ret_val;               /* value to return to caller */

#ifdef PARANOID
  if( !op_s ) {
    PANIC( "LOMAC: null op_s pointer passed to mediate_subject_object()\n" );
  }
  if( !p_subject ) {
    PANIC( "LOMAC: null subject pointer passed "
	   "to mediate_subject_object()\n" );
  }
  if( !p_object ) {
    PANIC( "LOMAC: null object pointer passed to mediate_subject_object()\n" );
  }
#endif

#ifdef NO_MEDIATION
  ret_val = 1;           /* allow operation regardless of decision */
#else
  ret_val = 0;           /* pessimistically assume deny */
#endif

  /* Get the levels of `p_subject' and `p_object' so we can compare them. */
  get_subject_lattr( p_subject, &subject_lattr );
  get_object_lattr(  p_object,  &object_lattr );

  /* If `p_subject's level is less than `p_object's level,      *
   * we must indicate that the operation should not be allowed. */
  if( LEVEL_LEQ( object_lattr.level, subject_lattr.level ) ) {
    
    ret_val = 1;         /* allow operation */

  } else {

    /* log permission denial */
    if( verbose & VERBOSE_DEMOTE_DENY ) {
      log_start();
      log_append_string( "LOMAC: level-" );
      log_append_int( subject_lattr.level );
      log_append_string( " subject " );
      log_append_subject_id( p_subject );
      log_append_string( " denied " );
      log_append_string( op_s );
      log_append_string( " to " );
      log_append_int( object_lattr.level );
      log_append_string( "-level object " );
      log_append_object_id( p_object );
      log_append_string( "\n" );
      log_print();
    }
    
  } /* if/else allow/deny */

#ifdef PARANOID
  if( !( ( ret_val == 0 ) || ( ret_val == 1 ) ) ) {
    PANIC( "LOMAC: bad return value in mediate_subject_object()\n" );
  }
#endif

  return( ret_val );

} /* mediate_subject_object() */


/* mediate_path_path()
 *
 * in:     op_s             - operation being mediated
 *         p_subject        - subject doing operation, for logging purposes
 *         canabsname_one_s - old name of file
 *         canabsname_two_s - proposed new name of file
 * out:    nothing
 * return: value   condition
 *         -----    ---------
 *           0      Caller should prevent operation
 *           1      Caller should permit operation
 * 
 * This function returns 1 if the level of `canabsname_two' is
 * less than or equal to the level of `canabsname_one'.  Otherwise,
 * it returns 0 and logs a permission denial on `op_s'.  The
 * two `canabsname' parameters must be paths in canonical absolute form.
 *
 * This function is used to mediate rename and link operations.
 *
 */
 
int mediate_path_path( const char *op_s, 
		       const lomac_subject_t *p_subject,
		       const char *canabsname_one_s,
		       const char *canabsname_two_s ) {

  lattr_t name_one_lattr;    /* lattr of `canabsname_one_s' */
  lattr_t name_two_lattr;    /* lattr of `canabsname_two_s' */
  int ret_val;               /* value to return to caller */

#ifdef PARANOID
  if( !op_s ) {
    PANIC( "LOMAC: null operation string passed to mediate_path_path()\n" );
  }
  if( !p_subject ) {
    PANIC( "LOMAC: null subject pointer passed to mediate_path_path()\n" );
  }
  if( !canabsname_one_s ) {
    PANIC( "LOMAC: null name one pointer passed to mediate_path_path()\n" );
  }
  if( !is_canabspath( canabsname_one_s ) ) {
    PANIC( "LOMAC: non-canonical/absolute name one "
	   "passed to mediate_path_path()\n" );
  }
  if( !canabsname_two_s ) {
    PANIC( "LOMAC: null name two passed to mediate_path_path()\n" );
  }
  if( !is_canabspath( canabsname_two_s ) ) {
    PANIC( "LOMAC: non-canonical/absolute name two "
	   "passed to mediate_path_path()\n" );
  }
#endif

#ifdef NO_MEDIATION
  ret_val = 1;           /* allow operation regardless of decision */
#else
  ret_val = 0;           /* pessimistically assume deny */
#endif

  /* Get levels so we can compare them. */
  get_pathname_lattr( canabsname_one_s, &name_one_lattr );
  get_pathname_lattr( canabsname_two_s, &name_two_lattr );

  /* If `canabsname_one_s' has a lower level than `canabsname_two_s', deny. */
  if( LEVEL_LEQ( name_two_lattr.level, name_one_lattr.level ) ) {

    ret_val = 1;         /* allow operation */

  } else {

    /* log permission denial */
    if( verbose & VERBOSE_DEMOTE_DENY ) {
      log_start();
      log_append_string( "LOMAC: subject " );
      log_append_subject_id( p_subject );
      log_append_string( " denied rename of " );
      log_append_int( name_one_lattr.level );
      log_append_string( "-level object " );
      log_append_string( canabsname_one_s );
      log_append_string( " to level-" );
      log_append_int( name_two_lattr.level );
      log_append_string( " name " );
      log_append_string( canabsname_two_s );
      log_append_string( "\n" );
      log_print();
    }

  } /* if/else allow/deny */
  
#ifdef PARANOID
  if( !( ( ret_val == 0 ) || ( ret_val == 1 ) ) ) {
    PANIC( "LOMAC: bad return value in mediate_path_path()\n" );
  }
#endif

  return( ret_val );

} /* mediate_path_path() */


/* mediate_subject_at_lattr()
 *
 * in:     op_s         - name of operation being mediated
 *         p_subject    - subject whose lattr we want to check
 *         target_lattr - lattr to compare to `p_subject's lattr
 *         
 * out:    nothing
 * return: value   condition
 *         -----   ---------
 *           0     `p_subject' is not at `target_lattr'
 *           1     `p_subject' is at `target_lattr'
 *
 * This function provides a predicate for determining whether or not
 * `p_subject' is at the lattr specified by `target_lattr'.  This
 * function compares `p_subject's lattr to `target_lattr'.  If the
 * lattrs match, it retruns 1.  Otherwise, it logs a permission denial
 * on `op_s' and returns 0.
 *
 */

int mediate_subject_at_lattr( const char *op_s, 
			      const lomac_subject_t *p_subject,
			      const lattr_t target_lattr ) {

  lattr_t subject_lattr;   /* lattr of `p_subject' */
  int ret_val;             /* value returned to caller */

#ifdef PARANOID
  if( !op_s ) {
    PANIC( "LOMAC: null operation name in mediate_subject_at_lattr().\n" );
  }
  if( !p_subject ) {
    PANIC( "LOMAC: null subject pointer in mediate_subject_at_lattr().\n" );
  }
  if( !( ( target_lattr.level == LOMAC_HIGHEST_LEVEL ) ||
	 ( target_lattr.level == LOMAC_LOWEST_LEVEL ) ) ) {
    PANIC( "LOMAC: invalid target lattr in mediate_subject_at_lattr().\n" );
  }
#endif

#ifdef NO_MEDIATION
  ret_val = 1;           /* allow operation regardless of decision */
#else
  ret_val = 0;           /* pessimistically assume deny */
#endif

  /* Make `subject_lattr' the lattr of `p_subject'. */
  get_subject_lattr( p_subject, &subject_lattr );

  /* compare with `target_lattr */
  if( LEVEL_LEQ( subject_lattr.level, target_lattr.level ) &&
      LEVEL_LEQ( target_lattr.level, subject_lattr.level ) ) {

    ret_val = 1;    /* allow operation */

  } else {

    /* log permission denial */
    if( verbose & VERBOSE_DEMOTE_DENY ) {
      log_start();
      log_append_string( "LOMAC: denied level-" );
      log_append_int( subject_lattr.level );
      log_append_string( " proc " );
      log_append_subject_id( p_subject );
      log_append_string( "'s " );
      log_append_string( op_s );
      log_append_string( ".\n" );
      log_print();
    }

  } /* if/else allow/deny */


#ifdef PARANOID
  if( !( ( ret_val == 0 ) || ( ret_val == 1 ) ) ) {
    PANIC( "LOMAC: bad return value in mediate_subject_at_lattr().\n" );
  }
#endif

  return( ret_val );

} /* mediate_subject_at_lattr() */


/* mediate_path_at_lattr()
 *
 * in:     op_s         - name of operation being mediated
 *         p_subject    - subject performing operation, for logging
 *         p_object     - object whose lattr we want to examine
 *         target_lattr - lattr to compare to `p_object's lattr
 *         
 * out:    nothing
 * return: value   condition
 *         -----   ---------
 *           0     `p_object' is not at `target_lattr'
 *           1     `p_object' is at `target_lattr'
 *
 * This function provides a predicate for determining whether or not
 * `p_object' is at the lattr specified by `target_lattr'.  This
 * function compares `p_object's lattr to `target_lattr'.  If the
 * lattrs match, it retruns 1.  Otherwise, it logs a permission denial
 * on `op_s' and returns 0.
 *
 */

int mediate_object_at_lattr( const char *op_s,
			     const lomac_subject_t *p_subject,
			     const lomac_object_t *p_object,
			     const lattr_t target_lattr ) {

  lattr_t object_lattr;  /* lattr of `p_object' */
  int ret_val;           /* value returned to caller */

#ifdef PARANOID
  if( !op_s ) {
    PANIC( "LOMAC: null operation name in mediate_object_at_lattr().\n" );
  }
  if( !p_subject ) {
    PANIC( "LOMAC: null subject pointer in mediate_object_at_lattr().\n" );
  }
  if( !p_object ) {
    PANIC( "LOMAC: null object pointer in mediate_object_at_lattr().\n" );
  }
  if( !( ( target_lattr.level == LOMAC_HIGHEST_LEVEL ) ||
	 ( target_lattr.level == LOMAC_LOWEST_LEVEL ) ) ) {
    PANIC( "LOMAC: invalid target lattr in mediate_object_at_lattr().\n" );
  }
#endif

#ifdef NO_MEDIATION
  ret_val = 1;           /* allow operation regardless of decision */
#else
  ret_val = 0;           /* pessimistically assume deny */
#endif

  /* Set `object_lattr' to the lattr of `p_object' */
  get_object_lattr( p_object, &object_lattr );

  /* compare with `target_lattr */
  if( LEVEL_LEQ( object_lattr.level, target_lattr.level ) &&
      LEVEL_LEQ( target_lattr.level, object_lattr.level ) ) {

    ret_val = 1;    /* allow operation */

  } else {

    /* log permission denial */
    if( verbose & VERBOSE_DEMOTE_DENY ) {
      log_start();
      log_append_string( "LOMAC: denied proc " );
      log_append_subject_id( p_subject );
      log_append_string( "'s " );
      log_append_string( op_s );
      log_append_string( " to level-" );
      log_append_int( object_lattr.level );
      log_append_string( " object " );
      log_append_object_id( p_object );
      log_append_string( ".\n" );
      log_print();
    }

  } /* if/else allow/deny */


#ifdef PARANOID
  if( !( ( ret_val == 0 ) || ( ret_val == 1 ) ) ) {
    PANIC( "LOMAC: bad return value in mediate_object_at_lattr().\n" );
  }
#endif

  return( ret_val );

} /* mediate_object_at_lattr() */

