/*****************************************************************
 *
 * lomac_monitor.c
 *
 * LOMAC - Low Water-Mark Mandatory Access Control for Linux
 * Copyright (c) 1999, 2000, 2001, 2002 Networks Associates
 * Technology, Inc.  All rights reserved.
 * 
 * This file is part of LOMAC.
 *
 * LOMAC is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * LOMAC is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with LOMAC; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *
 * This file contains functions which update LOMAC's internal
 * state in response to system events, such as successful
 * system calls.  These updates allow LOMAC to keep an accurate
 * picture of the kernel's state, enabling LOMAC to make reasonable
 * decisions when it mediates processes' use of security-relevant
 * system calls.  These functions perform no mediation themselves -
 * that is, they do not make access control decisions concerning
 * whether a given system call should be allowed or denied. This
 * mediation is handled by the functions in lomac_mediate.c.
 *
 *******************************************************************/

#include "kernel_interface.h"
#include "lomac_monitor.h"
#include "lomac_plm.h"
#include "lomac_tpl.h"
#include "lomac_log.h"

#ifdef PARANOID
/* probably need to remove this - it's not part of kernel interface. */
#include "kernel_util.h"  /* for IS_PIPE(), etc. */
#endif



/* monitor_open()
 *
 * in:     p_subject - subject that is opening `p_object'.
 *         p_object  - object being opened.
 * out:    nothing
 * return: nothing
 *
 * This function sets the level of objects as they are brought into memory
 * by the open/creat activity of processes.  In this version of LOMAC,
 * the proper level of any object (newly created or not) is determined
 * solely according to the PLM.  This function performs no mediation.
 *
 */

void monitor_open( lomac_subject_t *p_subject, lomac_object_t *p_object ) {

  lattr_t object_lattr = LATTR_INVALID;   /* lattr of `p_object' */

#ifdef PARANOID
  if( !p_subject ) {
    PANIC( "LOMAC: null subject pointer in monitor_open.\n" );
  }
  if( !p_object ) {
    PANIC( "LOMAC: null object pointer in monitor_open.\n" );
  }
#endif

  /* Optimization - check to see if `p_object' already has its level *
   * set, and if it does, don't bother to set it again.              */
  get_object_lattr( p_object, &object_lattr );
  if( object_lattr.level != LOMAC_INVALID_LEVEL ) {
    return;
  }


  if( get_object_canabspath( p_object, log_s, LOG_BUFFER_LENGTH ) ) {

#ifdef PARANOID
    /* Error - we are trying to get the level of a nameless object. */
    log_start();
    log_append_string( "LOMAC: unnamed object " );
    log_append_object_id( p_object );
    log_append_string( " in open!\n" );
    log_print();
    PANIC( "LOMAC: open with nameless object, see log\n" );
#endif

  } 

  get_pathname_lattr( log_s, &object_lattr );
  set_object_lattr( p_object, object_lattr );

  if( verbose & VERBOSE_LOG_OBJECTS ) {
    log_start();
    log_append_string( "LOMAC: " );
    log_append_subject_id( p_subject );
    log_append_string( " opens " );
    log_append_object_id( p_object );
    log_append_string( " level " );
    log_append_int( object_lattr.level );
    log_append_string( "\n" );
    log_print();
  }

#ifdef PARANOID
  get_object_lattr( p_object, &object_lattr );
  if( object_lattr.level == LOMAC_INVALID_LEVEL ) {
    PANIC( "LOMAC: monitor_open left object with invalid level.\n" );
  }
#endif

} /* monitor_open() */



/* monitor_read_object()
 *
 * in:     p_subject - subject that read `p_object'.
 *         p_object  - object read by `p_subject'.
 * out:    nothing
 * return: nothing
 *
 * This function examines the objects read by subjects.  If a subject
 * reads from an object with a level lower than its own, this function
 * reduces the subject's level to match the object's.  This lowering
 * is referred to as "demotion" in much of the LOMAC documentation.
 * This function performs no mediation.
 *
 * This function is for the following kinds of objects:
 *   files, FIFOs, and UNIX (local) domain sockets.
 * It is also used to monitor reads on unnamed pipes.  LOMAC does not
 * consider unnamed pipes to be objects, and treats them differently
 * from objects such as files.  However, the differences are mostly
 * in the write-handling behavior.  LOMAC's read-handling behavior is
 * the same both for objects and unnamed pipes, so this function
 * handles both cases.
 *
 * This function logs only on demotion.  Note that it would be unwise
 * to log every read instead, because every time this LKM prints a log
 * message, the kernel log daemon performs a read from /dev/kmem,
 * which need its own log message, leading to an infinite loop.
 */

void monitor_read_object( lomac_subject_t *p_subject,
			  lomac_object_t *p_object ) {

  lattr_t subject_lattr;     /* lattr of `p_subject' */
  lattr_t object_lattr;      /* lattr of `p_object' */

#ifdef PARANOID
  if( !p_subject ) {
    PANIC( "LOMAC: null subject pointer passed to monitor_read()\n" );
  }
  if( !p_object ) {
    PANIC( "LOMAC: null object pointer passed to monitor_read()\n" );
  }
#endif

  /* Get the lattrs of `p_subject' and `p_object' so we can compare them. */
  get_subject_lattr( p_subject, &subject_lattr );
  get_object_lattr(  p_object,  &object_lattr );

  /* If `p_object's level is less than `p_subject's level, *
   * we must demote `p_subject'.                           */
  while( ! LEVEL_LEQ( subject_lattr.level, object_lattr.level ) ) {

#ifdef TRUST
    /* If `p_subject' is running a trusted program, don't demote. */
    if( is_running_trusted_program( p_subject ) ) {

      if( verbose & VERBOSE_TRUST ) {
	log_start();
	log_append_string( "LOMAC: level-" );
	log_append_int( subject_lattr.level );
	log_append_string( " trusted subject " );
	log_append_subject_id( p_subject );
	log_append_string( " avoided demotion despite reading level-" );
	log_append_int( object_lattr.level );
	log_append_string( " object " );
	log_append_object_id( p_object );
	log_append_string( "\n" );
	log_print();
      }

      break; /* avoid demotion */
    }
#endif

    set_subject_lattr( p_subject, object_lattr );  /* demote! */
    if( verbose & VERBOSE_DEMOTE_DENY ) {
      log_start();
      log_append_string( "LOMAC: level-" );
      log_append_int( subject_lattr.level );
      log_append_string( " subject " );
      log_append_subject_id( p_subject );
      log_append_string( " demoted to level " );
      log_append_int( object_lattr.level );
      log_append_string( " after reading " );
      log_append_object_id( p_object );
      log_append_string( "\n" );
      log_print();
    }

#ifdef PARANOID
    get_subject_lattr( p_subject, &subject_lattr );
    get_object_lattr( p_object, &object_lattr );
    if( !( LEVEL_LEQ( subject_lattr.level, object_lattr.level ) &&
	   LEVEL_LEQ( object_lattr.level, subject_lattr.level ) ) ) {
      PANIC( "LOMAC: subject and object levels not equal after demotion\n" );
    }
#endif

    break;  /* we're using while to implement if */
  } /* if (implemented with while) we need to demote */

} /* monitor_read_object() */


/* monitor_read_nic()
 *
 * in:     p_subject - subject that read from NIC.
 *         p_nic     - NIC that `p_subject' read from.
 * out:    nothing
 * return: nothing
 *
 * This function monitors reads on network sockets.  The level of a
 * network socket can change with each read.  On a given read, the
 * socket's level is the floor of the levels associated with the NICs
 * that supplied the data read.
 *
 * This function logs only on demotion.  Note that it would be unwise
 * to log every read instead, because every time this LKM prints a log
 * message, the kernel log daemon performs a read from /dev/kmem,
 * which need its own log message, leading to an infinite loop.
 *
 */

void monitor_read_nic( lomac_subject_t *p_subject, lomac_nic_t *p_nic ) {

  lattr_t subject_lattr;          /* lattr of `p_subject' */
  lattr_t nic_lattr;              /* lattr of `p_nic' */

  get_subject_lattr( p_subject, &subject_lattr );

  /* Presently, we're putting all NICs at LOMAC_LOWEST_LEVEL.  The 0.1 *
   * and 0.2 prototypes determined socket levels as advertized in the  *
   * description of this function; this functionality will eventually  *
   * be moved into this prototype.                                     */
  nic_lattr = LATTR_LOW;

  /* If `p_nic's lattr is less than `p_subject's lattr, *
   * we must demote `p_subject'.                        */
  while( ! LEVEL_LEQ( subject_lattr.level, nic_lattr.level ) ) {

#ifdef TRUST
    /* If `p_subject' is running a trusted program, don't demote. */
    if( is_running_trusted_program( p_subject ) ) {

      if( verbose & VERBOSE_TRUST ) {
	log_start();
	log_append_string( "LOMAC: level-" );
	log_append_int( subject_lattr.level );
	log_append_string( " trusted subject " );
	log_append_subject_id( p_subject );
	log_append_string( " avoided demotion despite reading from level-" );
	log_append_int( nic_lattr.level );
	log_append_string( " NIC.\n" );
	log_print();
      }

      break;
    }
#endif

    set_subject_lattr( p_subject, nic_lattr );  /* demote! */
    if( verbose & VERBOSE_DEMOTE_DENY ) {
      log_start();
      log_append_string( "LOMAC: level-" );
      log_append_int( subject_lattr.level );
      log_append_string( " subject " );
      log_append_subject_id( p_subject );
      log_append_string( " demoted to level " );
      log_append_int( nic_lattr.level );
      log_append_string( " after reading from the network\n" );
      log_print();
    }

#ifdef PARANOID
    get_subject_lattr( p_subject, &subject_lattr );
    nic_lattr = LATTR_LOW;
    if( !( LEVEL_LEQ( subject_lattr.level, nic_lattr.level ) &&
	   LEVEL_LEQ( nic_lattr.level, subject_lattr.level ) ) ) {
      PANIC( "LOMAC: subject and nic levels not equal after demotion\n" );
    }
#endif

    break;  /* we're using while to implement if */
  } /* if (implemented with while) we need to demote */

} /* monitor_read_nic() */



/***********************************************************************
 * The following two routines to allow LOMAC to handle unnamed pipes.
 * LOMAC does not consider unnamed pipes to be objects, and constrains
 * their use as described below:
 * 
 * Originally (v0.3pre3), LOMAC enforced a simple set of constraints
 * that prevented subjects from using unnamed pipes as a conduit for
 * transferring low-intergrity data from low-integrity subjects (jobs)
 * to high-integrity subjects (jobs).  These constraints did not
 * involve assigning levels to unnamed pipes.  LOMAC ensured that a
 * given unnamed pipe was used only by processes in the same job.
 * LOMAC implemented this constraint by marking each pipe with the
 * pgrp number of the first process to access (read or write) it.
 * LOMAC denied subsequent accesses from processes whose pgrp numbers
 * didn't match the first.  This was the scheme described in: Fraser,
 * "LOMAC: Low Water-Mark Integrity Protection for COTS Environments,"
 * 2000 IEEE Symposium on Security and Privacy.
 * 
 * This original scheme worked fine for pipes used in shell pipelines,
 * but caused unacceptable failures in pipes used in some other
 * circumstances, particularly with cron.  (cron/logrotate and
 * cron/procmail were good examples.)  Several children of cron seemed
 * to enjoy sending messages to their parents using unnamed pipes
 * across job boundaries.  Consequently, this original unnamed pipe
 * constraint scheme has been replaced.
 *
 * The current version of LOMAC assigns levels to unnamed pipes.
 * However, unlike the levels it assigns to objects such as files,
 * sockets, and FIFOs, LOMAC allows the levels of unnamed pipes to
 * change over time.  When a subject creates a pipe, the pipe inherits
 * the subject's level (implemented in monitor_pipe_create(), below.)
 * When a low-level subject writes to a high-level pipe, LOMAC reduces
 * the pipe's level to match the level of the writing subject
 * (implemented in monitor_pipe_write(), below.)  When a high-level
 * subject reads from a low-level pipe, LOMAC reduces the reading
 * subject's level to match the pipe's (implemented by the same
 * monitor_read() code used for other types of objects).  LOMAC never
 * prevents a subject from writing to a pipe.
 *
 ***********************************************************************/


/* monitor_pipe_create()
 *
 * in:     p_subject - the subject that created `p_pipe'.
 *         p_pipe    - a newly-created pipe.
 * out:    p_pipe    - the pipe will be marked with the highest level.
 * return: nothing
 *
 *     This function marks `p_pipe' with the the same level as `p_subject'.
 *
 */

void monitor_pipe_create( lomac_subject_t *p_subject, 
			  lomac_object_t *p_pipe ) {

  lattr_t subject_lattr;       /* lattr of `p_subject' */

#ifdef PARANOID
  if( !p_subject ) {
    PANIC( "LOMAC/K: null subject pointer passed to monitor_pipe_create().\n");
  }
  if( !p_pipe ) {
    PANIC( "LOMAC/K: null pipe pointer passed to monitor_pipe_create().\n" );
  }
  if( !IS_PIPE(p_pipe) ) {
    PANIC( "LOMAC/K: monitor_pipe_create() given non-pipe.\n" );
  }
#endif

  get_subject_lattr( p_subject, &subject_lattr );
  set_object_lattr( p_pipe, subject_lattr );
  if( verbose & VERBOSE_LOG_OBJECTS ) {
    log_start();
    log_append_string( "LOMAC: " );
    log_append_subject_id( p_subject );
    log_append_string( " creates level-" );
    log_append_int( subject_lattr.level );
    log_append_string( " pipe " );
    log_append_object_id( p_pipe );
    log_append_string( "\n" );
    log_print();
  }

} /* monitor_pipe_create() */



/* monitor_pipe_write()
 *
 * in:     p_subject - subject that just wrote to `p_pipe'.
 *         p_pipe - pipe `p_subject' has written to.
 * out:    p_pipe - pipe may have its level adjusted.
 * return: nothing
 *
 *     This function should be called after a successful write to
 * `p_pipe'.  If the level of `p_subject' is less than the level of
 * `p_pipe', this function reduces `p_pipe's level to match
 * `current's.
 *
 *
 */

void monitor_pipe_write( lomac_subject_t *p_subject, lomac_object_t *p_pipe ) {

  lattr_t pipe_lattr;          /* lattr of `p_pipe' */
  lattr_t subject_lattr;       /* lattr of `p_subject' */

#ifdef PARANOID
  if( !p_subject ) {
    PANIC( "LOMAC/K: null subject pointer passed to monitor_pipe_write().\n" );
  }
  if( !p_pipe ) {
    PANIC( "LOMAC/K: null pipe pointer passed to monitor_pipe_write().\n" );
  }
  if( !IS_PIPE(p_pipe) ) {
    PANIC( "LOMAC/K: monitor_pipe_write() given non-pipe.\n" );
  }
#endif

  get_subject_lattr( p_subject, &subject_lattr );
  get_object_lattr( p_pipe, &pipe_lattr );
  if( !LEVEL_LEQ( pipe_lattr.level, subject_lattr.level ) ) {
    set_object_lattr( p_pipe, subject_lattr );
    if( verbose & VERBOSE_PIPE ) {
      log_start();
      log_append_string( "LOMAC: level-" );
      log_append_int( subject_lattr.level );
      log_append_string( " subject " );
      log_append_subject_id( p_subject );
      log_append_string( " contaminated level-" );
      log_append_int( pipe_lattr.level );
      log_append_string( " pipe " );
      log_append_object_id( p_pipe );
      log_append_string( "\n" );
      log_print();
    }
  }

} /* monitor_pipe_write() */


/* monitor_unix_socket_bind()
 *
 * in:     p_subject - subject doing the BIND operation on `p_socket'.
 *         p_socket  - socket server will use to solicit client connections.
 *         p_name    - `p_socket' is bound to this name in the filesystem.
 * out:    nothing
 * return: nothing
 *
 * LOMAC assigns levels to objects in the filesystem, including filesystem
 * objects that are the names of UNIX (local) domain sockets.  Servers
 * advertize their service by binding a name to their UNIX (local) domain
 * server socket using the BIND operation.  Subsequently, clients may
 * attempt to CONNECT to this server socket.  Since the CONNECT operation
 * operates on the server socket, and not its name in the filesystem, we
 * must ensure that the server socket has the same level as its name in
 * the filesystem.  We do this at BIND time with this function.
 *
 */

void monitor_unix_socket_bind( lomac_subject_t *p_subject,
			       lomac_object_t *p_socket, 
			       lomac_object_t *p_name ) {

  lattr_t name_lattr;    /* lattr of `p_name' */

#ifdef PARANOID
  lattr_t socket_lattr;   /* lattr of `p_socket' */

  if( !p_subject ) {
    PANIC( "LOMAC: null subject pointer in monitor_unix_socket_bind.\n" );
  }
  if( !p_socket ) {
    PANIC( "LOMAC: null socket pointer in monitor_unix_socket_bind.\n" );
  }
  if( !( IS_LOCAL_SOCKET( p_socket ) ) ) {
    PANIC( "LOMAC: monitor_unix_socket_bind given non-UNIX domain socket.\n" );
  }
  if( !p_name ) {
    PANIC( "LOMAC: null new socket pointer in monitor_unix_socket_bind.\n" );
  }

  /* don't try this check - get_object_level prints a warning *
   * when it is used to get LOMAC_INVALID_LEVEL.              *
   * get_object_level( p_socket, &socket_level );             *
   *  if( socket_level != LOMAC_INVALID_LEVEL ) {             */
#endif /* PARANOID */

  /* The BIND operation effectively creates a new object in the *
   * filesystem, specifically `p_name'.  First, we must set     *
   * `p_name's lattr.                                           */
  monitor_open( p_subject, p_name );

#ifdef PARANOID
  get_object_lattr( p_name, &name_lattr );
  if( name_lattr.level == LOMAC_INVALID_LEVEL ) {
    PANIC( "LOMAC: name lattr unset in monitor_unix_socket_bind.\n" );
  }
#endif

  /* Second, we must set p_socket's lattr to the same lattr as `p_name'. */
  get_object_lattr( p_name, &name_lattr );
  set_object_lattr( p_socket, name_lattr );
  
  if( verbose & VERBOSE_LOG_SOCKETS ) {
    log_start();
    log_append_string( "LOMAC: " );
    log_append_subject_id( p_subject );
    log_append_string( " binds " );
    log_append_object_id( p_socket );
    log_append_string( " to " );
    log_append_object_id( p_name );
    log_append_string( " level " );
    log_append_int( name_lattr.level );
    log_append_string( "\n" );
    log_print();
  }

#ifdef PARANOID
  get_object_lattr( p_socket, &socket_lattr );
  get_object_lattr( p_name, &name_lattr );
  if( socket_lattr.level != name_lattr.level ) {
    PANIC( "LOMAC: socket/name level mismatch in "
	   "monitor_unix_socket_bind.\n" );
  }
  if( socket_lattr.level == LOMAC_INVALID_LEVEL ) {
    PANIC( "LOMAC: socket left with invalid  level by "
	   "monitor_unix_socket_bind.\n" );
  }
#endif /* PARANOID */

} /* monitor_unix_socket_bind() */


/* monitor_unix_socket_abstract()
 *
 * in:     p_subject - subject doing  operation on `p_socket'.
 *         p_socket  - socket being operated upon
 * out:    nothing
 * return: nothing
 *
 * Linux 2.2 can bind unix domain sockets to names in an abstract
 * namespace separate from the traditional filesystem namespace.  It's
 * hard for LOMAC to assign levels to these sockets and those other
 * sockets that connect to them, because LOMAC likes to assign levels
 * to dentry structs, and abstract name bindings have no dentry
 * structs.
 *
 * For now, all UNIX-domain sockets bound to abstract names are set to
 * LOMAC_LOWEST_LEVEL.  This isn't a very good solution, because it
 * effectively prevents high-level processes from making use of these
 * sockets.  Sockets that connect to these abstract-bond sockets will
 * also be set to LOMAC_LOWEST_LEVEL.  This function can be used to do
 * all the setting.
 *
 * */

void monitor_unix_socket_abstract( lomac_subject_t *p_subject,
				   lomac_object_t *p_socket ) { 

#ifdef PARANOID
  if( !p_subject ) {
    PANIC( "LOMAC: null subject pointer in "
	   "monitor_unix_socket_bind_abstract.\n" );
  }
  if( !p_socket ) {
    PANIC( "LOMAC: null socket pointer in "
	   "monitor_unix_socket_bind_abstract.\n" );
  }
  if( !( IS_LOCAL_SOCKET( p_socket ) ) ) {
    PANIC( "LOMAC: monitor_unix_socket_bind_abstract "
	   "given non-UNIX domain socket.\n" );
  }

  /* don't try this check - get_object_level prints a warning *
   * when it is used to get LOMAC_INVALID_LEVEL.              *
   * get_object_level( p_socket, &socket_level );             *
   *  if( socket_level != LOMAC_INVALID_LEVEL ) {             */
#endif /* PARANOID */

  set_object_lattr( p_socket, LATTR_LOW );
  
  if( verbose & VERBOSE_LOG_SOCKETS ) {
    log_start();
    log_append_string( "LOMAC: " );
    log_append_subject_id( p_subject );
    log_append_string( " binds " );
    log_append_object_id( p_socket );
    log_append_string( " to abstract name level " );
    log_append_int( LOMAC_LOWEST_LEVEL );
    log_append_string( "\n" );
    log_print();
  }

} /* monitor_unix_socket_abstract() */


/* monitor_unix_socketpair()
 *
 * in:     p_subject - subject that created UNIX-domain socketpair
 *         p_socket1 - first socket in pair
 *         p_socket2 - second socket in pair
 * out:    nothing
 * return: nothing
 *
 * LOMAC sets the levels of UNIX domain sockets created with
 * socketpair to match the level of the creating process.
 *
 * */

void monitor_unix_socketpair( lomac_subject_t *p_subject,
			      lomac_object_t *p_socket1,
			      lomac_object_t *p_socket2 ) { 

  lattr_t subject_lattr;         /* lattr of `p_subject' */

#ifdef PARANOID
  lattr_t socket_lattr;

  if( !p_subject ) {
    PANIC( "LOMAC: null subject pointer in "
	   "monitor_unix_socketpair.\n" );
  }
  if( !( p_socket1 && p_socket2 ) ) {
    PANIC( "LOMAC: null socket pointer in "
	   "monitor_unix_socketpair.\n" );
  }
  if( !( IS_LOCAL_SOCKET( p_socket1 ) &&
	 IS_LOCAL_SOCKET( p_socket2 ) ) ) {
    PANIC( "LOMAC: monitor_unix_socketpair "
	   "given non-UNIX domain socket.\n" );
  }

  /* don't try this check - get_object_level prints a warning *
   * when it is used to get LOMAC_INVALID_LEVEL.              *
   * get_object_level( p_socket, &socket_level );             *
   *  if( socket_level != LOMAC_INVALID_LEVEL ) {             */
#endif /* PARANOID */

  get_subject_lattr( p_subject, &subject_lattr );
  set_object_lattr( p_socket1, subject_lattr );
  set_object_lattr( p_socket2, subject_lattr );
  
  if( verbose & VERBOSE_LOG_SOCKETS ) {
    log_start();
    log_append_string( "LOMAC: " );
    log_append_subject_id( p_subject );
    log_append_string( " creates socketpair " );
    log_append_object_id( p_socket1 );
    log_append_string( " and " );
    log_append_object_id( p_socket2 );
    log_append_string( " at level " );
    log_append_int( subject_lattr.level );
    log_append_string( "\n" );
    log_print();
  }

#ifdef PARANOID
  get_object_lattr( p_socket1, &socket_lattr );
  if( !( LEVEL_LEQ( subject_lattr.level, socket_lattr.level ) &&
	 LEVEL_LEQ( socket_lattr.level, subject_lattr.level ) ) ) {
    PANIC( "LOMAC: monitor_unix_socketpair failed to set socket1 level\n" );
  }
  get_object_lattr( p_socket2, &socket_lattr );
  if( !( LEVEL_LEQ( subject_lattr.level, socket_lattr.level ) &&
	 LEVEL_LEQ( socket_lattr.level, subject_lattr.level ) ) ) {
    PANIC( "LOMAC: monitor_unix_socketpair failed to set socket2 level\n" );
  }
#endif /* PARANOID */

} /* monitor_unix_socketpair() */


/* monitor_unix_socket_accept_connect()
 *
 * in:    p_subject    - process doing ACCEPT or CONNECT.
 *        p_old_socket - the socket whose level is already correctly set.
 *                       For accept, this is the original socket used to
 *                       receive the incoming connection.  For connect,
 *                       this is the socket created at the server
 *                       by the accept operation.
 *        p_new_socket - the socket whose level needs to be set.  For
 *                       accept, this is the session socket created for the
 *                       connection.  For connect, this is the client's
 *                       socket.
 * out:    nothing
 * return: nothing
 *
 * When a server calls ACCEPT, the kernel creates a new session socket
 * based on the original one used to receive the connection request.
 * We must ensure that the new socket has the same level as the
 * original one.
 *
 * When a client calls CONNECT, the kernel connects its socket
 * with the session socket the server created with ACCEPT.  We must
 * ensure that the client's socket has the same level as the server's
 * session socket.
 *
 */

void monitor_unix_socket_accept_connect( lomac_subject_t *p_subject,
					 lomac_object_t *p_old_socket,
					 lomac_object_t *p_new_socket ) {

  lattr_t old_socket_lattr;   /* lattr of `p_old_socket' */

#ifdef PARANOID
  lattr_t new_socket_lattr;   /* lattr of `p_new_socket' */

  if( !p_subject ) {
    PANIC( "LOMAC: null subject pointer in "
	   "monitor_unix_socket_accept_connect.\n" );
  }
  if( !p_old_socket ) {
    PANIC( "LOMAC: null old socket pointer in "
	   "monitor_unix_socket_accept_connect.\n" );
  }

  /* Don't try the following test... sometimes `p_old_socket' *
   * is a named object in the filesystem, like "/dev/log".    *
   * if( !( IS_LOCAL_SOCKET( p_old_socket ) ) ) {             */

  if( !p_new_socket ) {
    PANIC( "LOMAC: null new socket pointer in "
	   "monitor_unix_socket_accept_connect.\n" );
  }
  if( !( IS_LOCAL_SOCKET( p_new_socket ) ) ) {
    PANIC( "LOMAC: monitor_unix_socket_accept_connect "
	   "given non-local new socket.\n" );
  }
  get_object_lattr( p_old_socket, &old_socket_lattr );
  if( old_socket_lattr.level == LOMAC_INVALID_LEVEL ) {
    PANIC( "LOMAC: old socket has invalid level in "
	   "monitor_unix_socket_accept_connect.\n" );
  }

  /* Don't try the following test - get_object_lattr() burps out *
   * a warning if it is used to get a LOMAC_INVALID_LEVEL level. *
   * get_object_level( p_new_socket, &new_socket_level );        *
   * if( new_socket_level != LOMAC_INVALID_LEVEL ) {             */
#endif /* PARANOID */
  
  get_object_lattr( p_old_socket, &old_socket_lattr );
  set_object_lattr( p_new_socket, old_socket_lattr );

  if( verbose & VERBOSE_LOG_SOCKETS ) {
    log_start();
    log_append_string( "LOMAC: " );
    log_append_subject_id( p_subject );
    log_append_string( " propagates level " );
    log_append_int( old_socket_lattr.level );
    log_append_string( " from " );
    log_append_object_id( p_old_socket );
    log_append_string( " to " );
    log_append_object_id( p_new_socket );
    log_append_string( "\n" );
    log_print();
  }

#ifdef PARANOID
  get_object_lattr( p_old_socket, &old_socket_lattr );
  get_object_lattr( p_new_socket, &new_socket_lattr );
  if( new_socket_lattr.level != old_socket_lattr.level ) {
    PANIC( "LOMAC: socket level mismatch in "
	   "monitor_unix_socket_accept_connect.\n" );
  }
  if( new_socket_lattr.level == LOMAC_INVALID_LEVEL ) {
   PANIC( "LOMAC: socket left with invalid level by "
	  "monitor_unix_socket_accept_connect.\n" );
  } 
#endif /* PARANOID */

} /* handle_unix_domain_accept_connect() */

