// 
// FILE:
// serial.C
//
// FUNCTION:
// The linSerial class provides a simple, basic framework
// for managing a serial port.  See the header file for 
// additional documentation.
//
// HISTORY:
// Copyright (c) 1995, 1997 Linas Vepstas
// Released under the  conditions  of  the  GNU General 
// Public License.

#include <errno.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <termios.h>
#include <unistd.h>

#include "serial.h"

// ============================================================

linSerial::linSerial (void) 
{
   filename = 0x0;

   // set the informational logging level
   loglevel = INFO;
   loglevel = TRACE;

   fd = -1;
   serialport = 0x0;
   istty = 0;
   speed = 0;
}

// ============================================================

void linSerial::SetDTR (void) {
   if (!istty) return;
   int set;
   ioctl (fd, TIOCMGET, &set);
   set |= TIOCM_DTR;
   ioctl (fd, TIOCMSET, &set);
}

// ============================================================

void linSerial::ClearDTR (void) {
   if (!istty) return;
   int set;
   ioctl (fd, TIOCMGET, &set);
   set &= ~TIOCM_DTR;
   ioctl (fd, TIOCMSET, &set);
}

// ============================================================

int linSerial::SetFlowControl (int hw) {
   if (!istty) return 0;

   SETSCOPE ("linSerial::SetFlowControl()");

   // flush any unwritten, unread data
   tcflush(fd, TCIOFLUSH);

   struct termios tset;
   int retval = tcgetattr(fd, &tset);
   if (-1 == retval) {
      int norr = errno;
      LOGERR  ((SEVERE, "unable to tcgetattr: "));
      LOGMORE ((SEVERE, "%d %s \n", norr, strerror (norr)));
      return norr;
   }

   // Hardware flow control uses the RTS (Ready To Send) 
   // and CTS (clear to Send) lines. 
   // Software flow control uses IXON|IXOFF
   if (hw) {
      tset.c_iflag &= ~ (IXON|IXOFF);
      tset.c_cflag |= CRTSCTS;
      tset.c_cc[VSTART] = _POSIX_VDISABLE;
      tset.c_cc[VSTOP] = _POSIX_VDISABLE;
   } else {
      tset.c_iflag |= IXON|IXOFF;
      tset.c_cflag &= ~CRTSCTS;
      tset.c_cc[VSTART] = 0x11;           // 021 ^q
      tset.c_cc[VSTOP]  = 0x13;           // 023 ^s
   }
   
   retval = tcsetattr(fd, TCSANOW, &tset);
   if (-1 == retval) {
      int norr = errno;
      LOGERR  ((SEVERE, "unable to tcsetattr: "));
      LOGMORE ((SEVERE, "%d %s \n", norr, strerror (norr)));
      return norr;
   }
   
   return 0;
}

// ============================================================

int linSerial::SetBits (int bits, char parity, int stopbits) {
   if (!istty) return 0;

   SETSCOPE ("linSerial::SetBits()");

   // flush any unwritten, unread data
   tcflush(fd, TCIOFLUSH);

   struct termios tset;
   int retval = tcgetattr(fd, &tset);
   if (-1 == retval) {
      int norr = errno;
      LOGERR  ((SEVERE, "unable to tcgetattr: "));
      LOGMORE ((SEVERE, "%d %s \n", norr, strerror (norr)));
      return norr;
   }

   // set bits
   if (8 == bits) {
      tset.c_cflag &= ~CSIZE;
      tset.c_cflag |= CS8;
      tset.c_iflag &= ~ISTRIP;
   } else
   if (7 == bits) {
      tset.c_cflag &= ~CSIZE;
      tset.c_cflag |= CS7;
      tset.c_iflag |= ISTRIP;
   } else
   if (6 == bits) {
      tset.c_cflag &= ~CSIZE;
      tset.c_cflag |= CS6;
      tset.c_iflag |= ISTRIP;
   } else
   if (5 == bits) {
      tset.c_cflag &= ~CSIZE;
      tset.c_cflag |= CS5;
      tset.c_iflag |= ISTRIP;
   } else {
      LOGERR ((SEVERE, "invalid number of bits: %d \n", bits));
   }

   // set parity
   if ('N' == parity) {
      tset.c_cflag &= ~PARENB;
      tset.c_iflag &= ~INPCK;
   } else 
   if ('E' == parity) {
      tset.c_cflag |= PARENB;
      tset.c_cflag &= ~PARODD;
      tset.c_iflag |= INPCK;
   } else 
   if ('O' == parity) {
      tset.c_cflag |= PARENB;
      tset.c_cflag |= PARODD;
      tset.c_iflag |= INPCK;
   } else {
      LOGERR ((SEVERE, "invalid parity: %c \n", parity));
   }

   // set number of stop bits
   if (2 == stopbits) {
      tset.c_cflag |= CSTOPB;
   } else
   if (1 == stopbits) {
      tset.c_cflag &= ~CSTOPB;
   } else {
      LOGERR ((SEVERE, "invalid number of stop bits: %d \n", stopbits));
   }

   retval = tcsetattr(fd, TCSANOW, &tset);
   if (-1 == retval) {
      int norr = errno;
      LOGERR  ((SEVERE, "unable to tcsetattr: "));
      LOGMORE ((SEVERE, "%d %s \n", norr, strerror (norr)));
      return norr;
   }
   
   return 0;
}

// ============================================================

int linSerial :: SetupDefault (void) {
   if (!istty) return 0;

   SETSCOPE ("linSerial::SetupDefault()");

   // flush any unwritten, unread data
   tcflush(fd, TCIOFLUSH);

   struct termios tset;

   // Setup the terminal for ordinary I/O 
   // e.g. as an ordinary modem or serial port printer.
   // 
   // Default config: hangup on close.
   // if the HUPCL bit set, closing the port will hang up 
   // the modem. (i.e. will lower the modem control lines)
   //
   // Default config: use hardware flow control
   // if CRTSCTS is set, use h/w flow control (IXON/OFF should not be set).
   // The best documentation for these calls can be found in
   // the Linux man pages for termios(3) and stty(1).
   //
   // Default config:
   // 8 bits, no parity, one stop bit, 115200 baud
   tset.c_cflag = CREAD|CS8|B115200|CRTSCTS|HUPCL;

   // Default config: ignore break, do not ignore parity
   tset.c_iflag = IGNBRK;

   // Default: no delay on carriage return, backspace, tab, etc.
   tset.c_oflag = CR0|NL0|TAB0|BS0|FF0|VT0;

   tset.c_cc[VEOF]   = _POSIX_VDISABLE;
   tset.c_cc[VEOL]   = _POSIX_VDISABLE;
   tset.c_cc[VERASE] = _POSIX_VDISABLE;
   tset.c_cc[VINTR]  = _POSIX_VDISABLE;
   tset.c_cc[VKILL]  = _POSIX_VDISABLE;
   tset.c_cc[VQUIT]  = _POSIX_VDISABLE;
   tset.c_cc[VSUSP]  = _POSIX_VDISABLE;
   tset.c_cc[VSTART] = _POSIX_VDISABLE;
   tset.c_cc[VSTOP]  = _POSIX_VDISABLE;
   
   int retval = tcsetattr(fd, TCSANOW, &tset);
   if (-1 == retval) {
      int norr = errno;
      LOGERR  ((SEVERE, "unable to tcsetattr: "));
      LOGMORE ((SEVERE, "%d %s \n", norr, strerror (norr)));
      return norr;
   }
}

// ============================================================

int linSerial :: SetupRaw (void) {
   if (!istty) return 0;

   SETSCOPE ("linSerial::SetupRaw()");

   // flush any unwritten, unread data
   tcflush(fd, TCIOFLUSH);

   struct termios tset;
   int retval = tcgetattr(fd, &tset);
   if (-1 == retval) {
      int norr = errno;
      LOGERR  ((SEVERE, "unable to tcgetattr: "));
      LOGMORE ((SEVERE, "%d %s \n", norr, strerror (norr)));
      return norr;
   }

   // Default config: ignore break, do not ignore parity
   tset.c_iflag |= IGNBRK;

   // Default: no delay on carriage return, backspace, tab, etc.
   tset.c_oflag &= ~ (NLDLY|CRDLY|TABDLY|BSDLY|VTDLY|FFDLY);
   tset.c_oflag |= NL0|CR0|TAB0|BS0|VT0|FF0;

   // disable canonical processing
   tset.c_lflag &= ~ICANON;
  
   tset.c_cc[VEOF]   = _POSIX_VDISABLE;
   tset.c_cc[VEOL]   = _POSIX_VDISABLE;
   tset.c_cc[VERASE] = _POSIX_VDISABLE;
   tset.c_cc[VINTR]  = _POSIX_VDISABLE;
   tset.c_cc[VKILL]  = _POSIX_VDISABLE;
   tset.c_cc[VQUIT]  = _POSIX_VDISABLE;
   tset.c_cc[VSUSP]  = _POSIX_VDISABLE;
   tset.c_cc[VSTART] = _POSIX_VDISABLE;
   tset.c_cc[VSTOP]  = _POSIX_VDISABLE;
   
   retval = tcsetattr(fd, TCSANOW, &tset);
   if (-1 == retval) {
      int norr = errno;
      LOGERR  ((SEVERE, "unable to tcsetattr: "));
      LOGMORE ((SEVERE, "%d %s \n", norr, strerror (norr)));
      return norr;
   }
}

// ============================================================

int linSerial :: SetupTerminal (void) {
   if (!istty) return 0;

   SETSCOPE ("linSerial::SetupTerminal()");

   // flush any unwritten, unread data
   tcflush(fd, TCIOFLUSH);

   struct termios tset;
   int retval = tcgetattr(fd, &tset);
   if (-1 == retval) {
      int norr = errno;
      LOGERR  ((SEVERE, "unable to tcgetattr: "));
      LOGMORE ((SEVERE, "%d %s \n", norr, strerror (norr)));
      return norr;
   }

   // Default config: ignore break, do not ignore parity
   tset.c_iflag |= IGNBRK;

   // Default: no delay on carriage return, backspace, tab, etc.
   tset.c_oflag &= ~ (NLDLY|CRDLY|TABDLY|BSDLY|VTDLY|FFDLY);
   tset.c_oflag |= NL0|CR0|TAB0|BS0|VT0|FF0;

   // Default: setup cannonical processing, echo special control chars
   tset.c_lflag = ICANON|IEXTEN|ECHO|ECHOE|ECHOK|ECHOCTL|ECHOKE;
  
   tset.c_cc[VEOF]   = 0x04;
   tset.c_cc[VEOL]   = 0x0a;
   tset.c_cc[VERASE] = 0x08;            // BS '\b'
   tset.c_cc[VINTR]  = 0x03;            // 0177    DEL
   tset.c_cc[VKILL]  = _POSIX_VDISABLE; // '\025'  NAK
   tset.c_cc[VQUIT]  = _POSIX_VDISABLE; // 034 FS ^|
   tset.c_cc[VSUSP]  = 0x1a;            // 026 ^z suspend
   tset.c_cc[VSTART] = 0x11;            // 021 ^q
   tset.c_cc[VSTOP]  = 0x13;            // 023 ^s
   
   retval = tcsetattr(fd, TCSANOW, &tset);
   if (-1 == retval) {
      int norr = errno;
      LOGERR  ((SEVERE, "unable to tcsetattr: "));
      LOGMORE ((SEVERE, "%d %s \n", norr, strerror (norr)));
      return norr;
   }
}

// ============================================================

int linSerial::SetSpeed (int spd) 
{
   SETSCOPE ("linSerial::SetSpeed()");
   if (!istty) {
      LOGERR ((SEVERE, "can't set speed on a non-terminal device\n"));
      return ENOTTY;
   }

   // Always round up, so that we can talk to the 
   // serial port at a speed equal or faster than 
   // requested.  
   // Flow control at excessive speeds is up to the user.
   if (460800 < spd) {
      LOGERR ((WARN, "requested speed %d is higher than supported max 460800 \n", spd));
      return EDOM;
   } else
   if (230400 < spd) {
      speed = 460800; 
   } else
   if (115200 < spd) {
      speed = 230400; 
   } else
   if (57600 < spd) {
      speed = 115200;
   } else 
   if (38400 < spd) {
      speed = 57600;
   } else 
   if (28800 < spd) {
      speed = 38400;
   } else 
   if (19200 < spd) {
      speed = 28800;
   } else 
   if (14400 < spd) {
      speed = 19200; 
   } else 
   if (12000 < spd) {
      speed = 14400;
   } else 
   if (9600 < spd) {
      speed = 12000;
   } else 
   if (7200 < spd) {
      speed = 9600;
   } else 
   if (4800 < spd) {
      speed = 7200;
   } else 
   if (2400 < spd) {
      speed = 4800;
   } else 
   if (1800 < spd) {
      speed = 2400;
   } else 
   if (1200 < spd) {
      speed = 1800;
   } else 
   if (600 < spd) {
      speed = 1200;
   } else 
   if (300 < spd) {
      speed = 600;
   } else 
   if (0 < spd) {
      speed = 300;
   } else { 
      LOGERR ((WARN, "bad speed: %d \n", speed));
      return EDOM;
   }

   struct termios trm;
   int retval = tcgetattr (fd, &trm);
   if (-1 == retval) {
      int norr = errno;
      LOGERR  ((SEVERE, "Cant get termios:"));
      LOGMORE ((SEVERE, " errno=%d %s \n", norr, strerror (norr)));
      return EIO;
   }

   // always round up, so that we can talk to 
   // serialport faster than it can transmit (it'll 
   // handle flow onctrol on its own).
   switch (speed) {
      case 230400:
         speed = 230400; 
         cfsetispeed (&trm, B230400);
         cfsetospeed (&trm, B230400);
         break;
      case 115200:
         speed = 115200; 
         cfsetispeed (&trm, B115200);
         cfsetospeed (&trm, B115200);
         break;
      case 57600:
         speed = 57600; 
         cfsetispeed (&trm, B57600);
         cfsetospeed (&trm, B57600);
         break;
      case 38400:
         speed = 38400; 
         cfsetispeed (&trm, B38400);
         cfsetospeed (&trm, B38400);
         break;
      case 28800:
         speed = 38400; 
         cfsetispeed (&trm, B38400);
         cfsetospeed (&trm, B38400);
         break;
      case 14400:
         speed = 19200; 
         cfsetispeed (&trm, B19200);
         cfsetospeed (&trm, B19200);
         break;
      case 12000:
         speed = 19200; 
         cfsetispeed (&trm, B19200);
         cfsetospeed (&trm, B19200);
         break;
      case 9600:
         speed = 9600;     
         cfsetispeed (&trm, B9600);
         cfsetospeed (&trm, B9600);
         break;
      case 7200:
         speed = 9600; 
         cfsetispeed (&trm, B9600);
         cfsetospeed (&trm, B9600);
         break;
      case 4800:
         speed = 4800; 
         cfsetispeed (&trm, B4800);
         cfsetospeed (&trm, B4800);
         break;
      case 2400:
         speed = 2400; 
         cfsetispeed (&trm, B2400);
         cfsetospeed (&trm, B2400);
         break;
      case 1800:
         speed = 1800; 
         cfsetispeed (&trm, B1800);
         cfsetospeed (&trm, B1800);
         break;
      case 1200:
         speed = 1200; 
         cfsetispeed (&trm, B1200);
         cfsetospeed (&trm, B1200);
         break;
      case 600:
         speed = 600; 
         cfsetispeed (&trm, B600);
         cfsetospeed (&trm, B600);
      case 300:
         speed = 300; 
         cfsetispeed (&trm, B300);
         cfsetospeed (&trm, B300);
         break;
      default:
         LOGERR ((WARN,"Bad speed: %d \n", speed));
         return EIO;
   }

   // OK -- set the speed
   tcsetattr (fd, TCSANOW, &trm);

   LOGERR ((INFO, "speed now %d \n", speed));
   return 0;
}

// ============================================================

int linSerial::OpenDevice (const char *filnam)
{
   SETSCOPE ("linSerial::OpenDevice()");

   // if there is another open device, close it first.
   if (serialport) {
      LOGERR ((INFO, "Closing previous port\n"));
      fflush (serialport);
      Close ();
   }

   LOGERR ((INFO, "opening device %s \n", filnam));
   if (!filnam) return 0;

   // make a copy of the filename
   if (filename) delete[] filename;
   filename = new char [strlen(filnam)+1];
   strcpy (filename, filnam);

   // check to see if this file exists already.  
   // If it does not, then the user is trying to 
   // create an ordinary file.
   struct stat buf;
   int retval = stat (filename, &buf);
   if (retval) {
      LOGERR ((INFO, "Attempting to create ordinary file %s \n", filename));
      serialport = fopen (filename, "a+");
      fd = fileno (serialport);
      istty = 0;
   } else {

      // open the serial port, make sure that its not the controlling tty
      // serialport = fopen ("/dev/modem", "r+");
      fd = open (filename, O_RDWR | O_NOCTTY);
      serialport = fdopen (fd, "a+");
   
      // if the device is e.g. a modem, and the modem is being used,
      // then busy-wait until it is free.
      while ((!serialport) && (EBUSY == errno)) {
         LOGERR ((WARN, "Serial Port busy, retry in 10 seconds\n"));
         sleep (10);
         fd = open (filename, O_RDWR | O_NOCTTY);
         serialport = fdopen (fd, "a+");
      }
      istty = 1;
   }

   // If we still failed, announce the bad news.
   if (!serialport) {
      int norr = errno;
      LOGERR  ((SEVERE, "Failed to open %s: ", filename));
      LOGMORE ((SEVERE, "%d %s \n", norr, strerror (norr)));
      return norr;
   }

   // do the following only if not an ordinary file.
   if (istty) {
      // Don't block on device configuration ...
      int flags = fcntl(fd, F_GETFL, 0);
      retval = fcntl(fd, F_SETFL, flags | O_NDELAY);
      if (-1 == retval) {
         int norr = errno;
         LOGERR  ((SEVERE, "Unable to set N_DELAY: "));
         LOGMORE ((SEVERE, "%d %s \n", norr, strerror (norr)));
         return norr;
      }
   
      // flush any garbage remaining on the port from previous operations.
      retval = tcflush (fd, TCIOFLUSH);
      if (-1 == retval) {
         int norr = errno;
   
         if (ENOTTY == norr) {
            istty = 0;
            LOGERR ((INFO, "Filename %s is not a tty \n", filename));
         } else {
            LOGERR  ((SEVERE, "unable to tcflush: "));
            LOGMORE ((SEVERE, "%d %s \n", norr, strerror (norr)));
            return norr;
         }
      }
   
      // configure the terminal
      SetupDefault ();
   	
      // allow commmunications to block
      flags = fcntl(fd, F_GETFL, 0);
      retval = fcntl(fd, F_SETFL, flags & ~O_NDELAY);
      if (-1 == retval) {
         int norr = errno;
         LOGERR  ((SEVERE, "Unable to unblock: "));
         LOGMORE ((SEVERE, "%d %s \n", norr, strerror (norr)));
         return norr;
      }

      // indicate that we are ready
      SetDTR ();
   }

   LOGERR ((INFO, "serialport opened on fd %d \n", fd));
}

// ============================================================

void linSerial::Close (void)
{
   SETSCOPE ("linSerial::Close()");

   if (0 >= fd) return;
   if (0x0 == serialport) return;

   LOGERR ((INFO, "closing serial port %s\n", filename));

   if (istty) {
      struct termios trm;
      int retval = tcgetattr (fd, &trm);
      if (-1 == retval) {
         int norr = errno;
         LOGERR  ((SEVERE, "couldn't get termios: "));
         LOGMORE ((SEVERE, "errno=%d %s \n", norr, strerror (norr)));
         fclose (serialport);
         serialport = 0x0;
         fd = -1;
         speed = 0;
         if (filename) delete [] filename;
         filename = 0x0;
         return;
      }

      // set terminal not to hangup on close
      trm.c_cflag &= ~HUPCL;
      tcsetattr (fd, TCSADRAIN, &trm);
   }
   
   fclose (serialport);
   serialport = 0x0;
   fd = -1;
   speed = 0;

   // free the filename
   if (filename) delete [] filename;
   filename = 0x0;

   LOGERR ((INFO, "serial port closed\n"));
}

// ====================== END OF FILE ============================
