/*
 * FILE:
 * gtkstripchartstrip.c
 *
 * FUNCTION:
 * Strip for the strip chart
 *
 * HISTORY:
 * Lins Vepstas February 2002
 */

#include <gdk/gdkkeysyms.h>

#include "gtkstripchartstrip.h"

/* ========================================================== */

/* class methods */
static void gtk_strip_chart_strip_class_init (GtkStripChartStripClass *);
static void gtk_strip_chart_strip_init (GtkStripChartStrip *);
static void gtk_strip_chart_strip_destroy (GtkObject *object);


/* ========================================================== */

GtkType	
gtk_strip_chart_strip_get_type (void)
{
  static GtkType strp_type = 0;

  if (!strp_type)
    {
      GtkTypeInfo strp_info =
      {
        "GtkStripChartStrip",
        sizeof (GtkStripChartStrip),
        sizeof (GtkStripChartStripClass),
        (GtkClassInitFunc) gtk_strip_chart_strip_class_init,
        (GtkObjectInitFunc) gtk_strip_chart_strip_init,
        /* reserved 1*/ NULL,
        /* reserved 2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      strp_type = gtk_type_unique (gtk_plot_get_type(), &strp_info);
    }
  return strp_type;

}

/* ========================================================== */

static void
gtk_strip_chart_strip_class_init (GtkStripChartStripClass *klass)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;

  object_class = (GtkObjectClass *) klass;
  widget_class = (GtkWidgetClass *) klass;

  // widget_class->draw = gtk_strip_chart_strip_draw;

  // widget_class->show_all = gtk_strip_chart_strip_show_all;

  object_class->destroy = gtk_strip_chart_strip_destroy;
}

/* ========================================================== */

static void
gtk_strip_chart_strip_init (GtkStripChartStrip *strip)
{
  GtkPlot *plot;

  strip->xpts = NULL;
  strip->ypts = NULL;
  strip->npts = 0;

  strip->curs_list = NULL;
  strip->limit_list = NULL;
  strip->plot_data_list = NULL;
  strip->stubs = NULL;

  /* -------------------------------- */
  /* drawing attributes */

  strip->override_symbol_color = 1;
  strip->override_symbol_border_color = 1;

  strip->line_attrs.line_style = GTK_PLOT_LINE_SOLID;
  strip->line_attrs.line_width = 2.0;
#if (0 == GTKEXTRA_MAJOR_VERSION) && \
    (99 == GTKEXTRA_MINOR_VERSION) && \
    (17 < GTKEXTRA_MICRO_VERSION)
  strip->line_attrs.cap_style = GDK_CAP_BUTT;
  strip->line_attrs.join_style = GDK_JOIN_MITER;
#endif

  strip->symbol.symbol_type = GTK_PLOT_SYMBOL_CIRCLE;
  strip->symbol.symbol_style = GTK_PLOT_SYMBOL_FILLED; 
  strip->symbol.size = 4;
  strip->symbol.border.line_width = 1;

  strip->line_connector = GTK_PLOT_CONNECT_STRAIGHT;

  /* -------------------------------- */
  plot = GTK_PLOT(strip);
  // gtk_plot_canvas_set_active_plot(chart->canvas, plot);
  // ????
  // gtk_plot_set_magnification(plot, 2.0);

  // gtk_plot_axis_set_title (plot, GTK_PLOT_AXIS_LEFT, "Thickness (Inches)");
  gtk_plot_axis_hide_title (plot, GTK_PLOT_AXIS_TOP);
  gtk_plot_axis_hide_title (plot, GTK_PLOT_AXIS_BOTTOM);
  gtk_plot_axis_hide_title (plot, GTK_PLOT_AXIS_RIGHT);

  gtk_plot_hide_legends (plot);

  gtk_plot_axis_show_labels (plot, GTK_PLOT_AXIS_TOP, 0);
  gtk_plot_axis_show_ticks (plot, GTK_PLOT_AXIS_TOP, 0, 0);

  gtk_plot_axis_show_labels (plot, GTK_PLOT_AXIS_BOTTOM, 0);
  gtk_plot_axis_show_ticks (plot, GTK_PLOT_AXIS_BOTTOM, 0, 0);

}

/* ========================================================== */

GtkWidget *
gtk_strip_chart_strip_new (void)
{
  GtkStripChartStrip *strp;

  strp = gtk_type_new (gtk_strip_chart_strip_get_type ());

  return GTK_WIDGET (strp);
}

/* ========================================================== */

static void 
gtk_strip_chart_strip_destroy (GtkObject *object)
{
  GList *node;
  GtkStripChartStrip *strip;
  GtkPlot *plot;

  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_STRIP_CHART_STRIP (object));

  strip = GTK_STRIP_CHART_STRIP (object);
  plot = GTK_PLOT (object);
  for (node = strip->plot_data_list; node; node=node->next)
  {
     GtkPlotData *pld = node->data;
     gtk_plot_remove_data (plot, pld);
     gtk_object_destroy (GTK_OBJECT (pld));
  }
  g_free (strip->plot_data_list);

  
  for (node = strip->curs_list; node; node=node->next)
  {
     GtkPlotData *pld = node->data;
     gtk_plot_remove_data (plot, pld);
     gtk_object_destroy (GTK_OBJECT (pld));
  }
  g_free (strip->curs_list);

  for (node = strip->limit_list; node; node=node->next)
  {
     GtkPlotData *pld = node->data;
     gtk_plot_remove_data (plot, pld);
     gtk_object_destroy (GTK_OBJECT (pld));
  }
  g_free (strip->limit_list);

  for (node = strip->stubs; node; node=node->next)
  {
     g_free (node->data);
  }
  g_free (strip->stubs);

}

/* ========================================================== */

void
gtk_strip_chart_strip_add_cursor (GtkStripChartStrip *strip, 
                                  GtkStripChartCursor *curs)
{
  GtkPlot *plot;
  GtkPlotData *pld;
  gdouble ymin, ymax;

  g_return_if_fail (strip != NULL);
  g_return_if_fail (GTK_IS_STRIP_CHART_STRIP (strip));

  g_return_if_fail (curs != NULL);
  g_return_if_fail (GTK_IS_STRIP_CHART_CURSOR (curs));

  plot = GTK_PLOT (strip);
  pld = GTK_PLOT_DATA (curs);

  /* make sure the height of the cursor fits the graph nicely */
  gtk_plot_get_yrange (plot, &ymin, &ymax);
  gtk_strip_chart_cursor_set_height (curs, ymin, ymax);

  gtk_plot_add_data (plot, pld);
  gtk_widget_show (GTK_WIDGET(curs));

  strip->curs_list = g_list_append (strip->curs_list, curs);

  curs->strip = plot;  /* backpointer to self */
}

/* ========================================================== */

static gint 
yorder (gconstpointer a, gconstpointer b)
{
  const GtkStripChartLimit *la = a;
  const GtkStripChartLimit *lb = b;
  return (la->range_max < lb->range_max);
}

void
gtk_strip_chart_strip_add_limit (GtkStripChartStrip *strip, 
                                 GtkStripChartLimit *lim)
{
  GtkPlot *plot;
  GtkPlotData *pld;
  gdouble xmin, xmax;

  g_return_if_fail (strip != NULL);
  g_return_if_fail (GTK_IS_STRIP_CHART_STRIP (strip));

  g_return_if_fail (lim != NULL);
  g_return_if_fail (GTK_IS_STRIP_CHART_LIMIT (lim));

  plot = GTK_PLOT (strip);
  pld = GTK_PLOT_DATA (lim);

  /* make sure the height of the cursor fits the graph nicely */
  gtk_plot_get_xrange (plot, &xmin, &xmax);
  gtk_strip_chart_limit_set_width (lim, xmin, xmax);

  gtk_plot_add_data (plot, pld);
  gtk_widget_show (GTK_WIDGET(lim));

  strip->limit_list = g_list_insert_sorted (strip->limit_list, 
        lim, yorder);

}

/* ========================================================== */

void
gtk_strip_chart_strip_set_xrange (GtkStripChartStrip *strip,
                                  gdouble xmin, gdouble xmax)
{
  GList *node;

  GtkPlot *plot;

  g_return_if_fail (strip != NULL);
  g_return_if_fail (GTK_IS_STRIP_CHART_STRIP (strip));

  plot = GTK_PLOT (strip);

  gtk_plot_set_xrange (plot, xmin, xmax);

  for (node = strip->limit_list; node; node=node->next)
  {
    GtkStripChartLimit *lim = node->data;
    gtk_strip_chart_limit_set_width (lim, xmin, xmax);
  }
}


/* ========================================================== */

void
gtk_strip_chart_strip_set_yrange (GtkStripChartStrip *strip,
                                  gdouble ymin, gdouble ymax)
{
  GList *node;

  GtkPlot *plot;

  g_return_if_fail (strip != NULL);
  g_return_if_fail (GTK_IS_STRIP_CHART_STRIP (strip));

  plot = GTK_PLOT (strip);

  gtk_plot_set_yrange (plot, ymin, ymax);

  for (node = strip->curs_list; node; node=node->next)
  {
    GtkStripChartCursor *curs = node->data;
    gtk_strip_chart_cursor_set_height (curs, ymin, ymax);
  }
}


/* ======================== END OF FILE ===================== */
